/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*-  vi:set ts=8 sts=4 sw=4: */

/*
    Sonic Visualiser
    An audio file viewer and annotation editor.
    Centre for Digital Music, Queen Mary, University of London.
    This file copyright 2006 Chris Cannam and QMUL.
    
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#include "Preferences.h"

#include "Exceptions.h"

#include "TempDirectory.h"

#include <QDir>
#include <QFileInfo>
#include <QMutex>
#include <QSettings>

Preferences *
Preferences::m_instance = 0;

Preferences *
Preferences::getInstance()
{
    if (!m_instance) m_instance = new Preferences();
    return m_instance;
}

Preferences::Preferences() :
    m_spectrogramSmoothing(SpectrogramZeroPadded),
    m_tuningFrequency(440),
    m_propertyBoxLayout(VerticallyStacked),
    m_windowType(HanningWindow),
    m_resampleQuality(1),
    m_omitRecentTemps(true),
    m_tempDirRoot(""),
    m_resampleOnLoad(false)
{
    QSettings settings;
    settings.beginGroup("Preferences");
    m_spectrogramSmoothing = SpectrogramSmoothing
        (settings.value("spectrogram-smoothing", int(m_spectrogramSmoothing)).toInt());
    m_tuningFrequency = settings.value("tuning-frequency", 440.f).toDouble();
    m_propertyBoxLayout = PropertyBoxLayout
        (settings.value("property-box-layout", int(VerticallyStacked)).toInt());
    m_windowType = WindowType
        (settings.value("window-type", int(HanningWindow)).toInt());
    m_resampleQuality = settings.value("resample-quality", 1).toInt();
    m_resampleOnLoad = settings.value("resample-on-load", false).toBool();
    m_backgroundMode = BackgroundMode
        (settings.value("background-mode", int(BackgroundFromTheme)).toInt());
    settings.endGroup();

    settings.beginGroup("TempDirectory");
    m_tempDirRoot = settings.value("create-in", "$HOME").toString();
    settings.endGroup();
}

Preferences::~Preferences()
{
}

Preferences::PropertyList
Preferences::getProperties() const
{
    PropertyList props;
    props.push_back("Spectrogram Smoothing");
    props.push_back("Tuning Frequency");
    props.push_back("Property Box Layout");
    props.push_back("Window Type");
    props.push_back("Resample Quality");
    props.push_back("Omit Temporaries from Recent Files");
    props.push_back("Resample On Load");
    props.push_back("Temporary Directory Root");
    props.push_back("Background Mode");
    return props;
}

QString
Preferences::getPropertyLabel(const PropertyName &name) const
{
    if (name == "Spectrogram Smoothing") {
        return tr("Spectrogram y-axis smoothing:");
    }
    if (name == "Tuning Frequency") {
        return tr("Frequency of concert A");
    }
    if (name == "Property Box Layout") {
        return tr("Property box layout");
    }
    if (name == "Window Type") {
        return tr("Spectral analysis window shape");
    }
    if (name == "Resample Quality") {
        return tr("Playback resampler type");
    }
    if (name == "Omit Temporaries from Recent Files") {
        return tr("Omit temporaries from Recent Files menu");
    }
    if (name == "Resample On Load") {
        return tr("Resample mismatching files on import");
    }
    if (name == "Temporary Directory Root") {
        return tr("Location for cache file directory");
    }
    if (name == "Background Mode") {
        return tr("Background colour preference");
    }
    return name;
}

Preferences::PropertyType
Preferences::getPropertyType(const PropertyName &name) const
{
    if (name == "Spectrogram Smoothing") {
        return ValueProperty;
    }
    if (name == "Tuning Frequency") {
        return RangeProperty;
    }
    if (name == "Property Box Layout") {
        return ValueProperty;
    }
    if (name == "Window Type") {
        return ValueProperty;
    }
    if (name == "Resample Quality") {
        return ValueProperty;
    }
    if (name == "Omit Temporaries from Recent Files") {
        return ToggleProperty;
    }
    if (name == "Resample On Load") {
        return ToggleProperty;
    }
    if (name == "Temporary Directory Root") {
        // It's an arbitrary string, we don't have a set of values for this
        return InvalidProperty;
    }
    if (name == "Background Mode") {
        return ValueProperty;
    }
    return InvalidProperty;
}

int
Preferences::getPropertyRangeAndValue(const PropertyName &name,
                                      int *min, int *max, int *deflt) const
{
    if (name == "Spectrogram Smoothing") {
        if (min) *min = 0;
        if (max) *max = 2;
        if (deflt) *deflt = int(SpectrogramZeroPadded);
        return int(m_spectrogramSmoothing);
    }

    //!!! freq mapping

    if (name == "Property Box Layout") {
        if (min) *min = 0;
        if (max) *max = 1;
        if (deflt) *deflt = 0;
        return m_propertyBoxLayout == Layered ? 1 : 0;
    }        

    if (name == "Window Type") {
        if (min) *min = int(RectangularWindow);
        if (max) *max = int(BlackmanHarrisWindow);
        if (deflt) *deflt = int(HanningWindow);
        return int(m_windowType);
    }

    if (name == "Resample Quality") {
        if (min) *min = 0;
        if (max) *max = 2;
        if (deflt) *deflt = 1;
        return m_resampleQuality;
    }

    if (name == "Omit Temporaries from Recent Files") {
        if (deflt) *deflt = 1;
    }

    if (name == "Background Mode") {
        if (min) *min = 0;
        if (max) *max = 2;
        if (deflt) *deflt = 0;
        return int(m_backgroundMode);
    }        

    return 0;
}

QString
Preferences::getPropertyValueLabel(const PropertyName &name,
                                   int value) const
{
    if (name == "Property Box Layout") {
        if (value == 0) return tr("Show boxes for all panes");
        else return tr("Show box for current pane only");
    }
    if (name == "Window Type") {
        switch (WindowType(value)) {
        case RectangularWindow: return tr("Rectangular");
        case BartlettWindow: return tr("Triangular");
        case HammingWindow: return tr("Hamming");
        case HanningWindow: return tr("Hanning");
        case BlackmanWindow: return tr("Blackman");
        case GaussianWindow: return tr("Gaussian");
        case ParzenWindow: return tr("Parzen");
        case NuttallWindow: return tr("Nuttall");
        case BlackmanHarrisWindow: return tr("Blackman-Harris");
        }
    }
    if (name == "Resample Quality") {
        switch (value) {
        case 0: return tr("Fastest");
        case 1: return tr("Standard");
        case 2: return tr("Highest quality");
        }
    }
    if (name == "Spectrogram Smoothing") {
        switch (value) {
        case NoSpectrogramSmoothing: return tr("None - blocky but accurate");
        case SpectrogramInterpolated: return tr("Interpolate - fast but fuzzy");
        case SpectrogramZeroPadded: return tr("Zero pad FFT - slow but clear");
        }
    }
    if (name == "Background Mode") {
        switch (value) {
        case BackgroundFromTheme: return tr("Follow desktop theme");
        case DarkBackground: return tr("Dark background");
        case LightBackground: return tr("Light background");
        }
    }
            
    return "";
}

QString
Preferences::getPropertyContainerName() const
{
    return tr("Preferences");
}

QString
Preferences::getPropertyContainerIconName() const
{
    return "preferences";
}

void
Preferences::setProperty(const PropertyName &name, int value) 
{
    if (name == "Spectrogram Smoothing") {
        setSpectrogramSmoothing(SpectrogramSmoothing(value));
    } else if (name == "Tuning Frequency") {
        //!!!
    } else if (name == "Property Box Layout") {
        setPropertyBoxLayout(value == 0 ? VerticallyStacked : Layered);
    } else if (name == "Window Type") {
        setWindowType(WindowType(value));
    } else if (name == "Resample Quality") {
        setResampleQuality(value);
    } else if (name == "Omit Temporaries from Recent Files") {
        setOmitTempsFromRecentFiles(value ? true : false);
    } else if (name == "Background Mode") {
        setBackgroundMode(BackgroundMode(value));
    }
}

void
Preferences::setSpectrogramSmoothing(SpectrogramSmoothing smoothing)
{
    if (m_spectrogramSmoothing != smoothing) {

        // "smoothing" is one of those words that looks increasingly
        // ridiculous the more you see it.  Smoothing smoothing smoothing.
        m_spectrogramSmoothing = smoothing;

        QSettings settings;
        settings.beginGroup("Preferences");
        settings.setValue("spectrogram-smoothing", int(smoothing));
        settings.endGroup();
        emit propertyChanged("Spectrogram Smoothing");
    }
}

void
Preferences::setTuningFrequency(float freq)
{
    if (m_tuningFrequency != freq) {
        m_tuningFrequency = freq;
        QSettings settings;
        settings.beginGroup("Preferences");
        settings.setValue("tuning-frequency", freq);
        settings.endGroup();
        emit propertyChanged("Tuning Frequency");
    }
}

void
Preferences::setPropertyBoxLayout(PropertyBoxLayout layout)
{
    if (m_propertyBoxLayout != layout) {
        m_propertyBoxLayout = layout;
        QSettings settings;
        settings.beginGroup("Preferences");
        settings.setValue("property-box-layout", int(layout));
        settings.endGroup();
        emit propertyChanged("Property Box Layout");
    }
}

void
Preferences::setWindowType(WindowType type)
{
    if (m_windowType != type) {
        m_windowType = type;
        QSettings settings;
        settings.beginGroup("Preferences");
        settings.setValue("window-type", int(type));
        settings.endGroup();
        emit propertyChanged("Window Type");
    }
}

void
Preferences::setResampleQuality(int q)
{
    if (m_resampleQuality != q) {
        m_resampleQuality = q;
        QSettings settings;
        settings.beginGroup("Preferences");
        settings.setValue("resample-quality", q);
        settings.endGroup();
        emit propertyChanged("Resample Quality");
    }
}

void
Preferences::setOmitTempsFromRecentFiles(bool omit)
{
    if (m_omitRecentTemps != omit) {
        m_omitRecentTemps = omit;
        QSettings settings;
        settings.beginGroup("Preferences");
        settings.setValue("omit-recent-temporaries", omit);
        settings.endGroup();
        emit propertyChanged("Omit Temporaries from Recent Files");
    }
}

void
Preferences::setTemporaryDirectoryRoot(QString root)
{
    if (root == QDir::home().absolutePath()) {
        root = "$HOME";
    }
    if (m_tempDirRoot != root) {
        m_tempDirRoot = root;
        QSettings settings;
        settings.beginGroup("TempDirectory");
        settings.setValue("create-in", root);
        settings.endGroup();
        emit propertyChanged("Temporary Directory Root");
    }
}

void
Preferences::setResampleOnLoad(bool resample)
{
    if (m_resampleOnLoad != resample) {
        m_resampleOnLoad = resample;
        QSettings settings;
        settings.beginGroup("Preferences");
        settings.setValue("resample-on-load", resample);
        settings.endGroup();
        emit propertyChanged("Resample On Load");
    }
}

void
Preferences::setBackgroundMode(BackgroundMode mode)
{
    if (m_backgroundMode != mode) {

        m_backgroundMode = mode;

        QSettings settings;
        settings.beginGroup("Preferences");
        settings.setValue("background-mode", int(mode));
        settings.endGroup();
        emit propertyChanged("Background Mode");
    }
}


