function solver = SMALL_solve(Problem, solver)
%% SMALL sparse solver
%
%   Function gets as input SMALL structure that contains SPARCO problem to
%   be solved, name of the toolbox and solver, and parameters file for
%   particular solver.
%
%   Outputs are solution, reconstructed signal and time spent

%   Centre for Digital Music, Queen Mary, University of London.
%   This file copyright 2009 Ivan Damnjanovic.
%
%   This program is free software; you can redistribute it and/or
%   modify it under the terms of the GNU General Public License as
%   published by the Free Software Foundation; either version 2 of the
%   License, or (at your option) any later version.  See the file
%   COPYING included with this distribution for more information.
%   
%%

if isa(Problem.A,'float')
    A = Problem.A;
    SparseLab_A=Problem.A;
    m = size(Problem.A,1);      % m is the no. of rows.
    n = size(Problem.A,2);      % n is the no. of columns.
else
    A  = @(x) Problem.A(x,1);  % The operator
    AT = @(y) Problem.A(y,2);  % and its transpose.
    SparseLab_A =@(mode, m, n, x, I, dim) SL_A(Problem.A, mode, m, n, x, I, dim);
    m = Problem.sizeA(1);      % m is the no. of rows.
    n = Problem.sizeA(2);      % n is the no. of columns.
    
end
% if signal that needs to be represented is different then training set for
% dictionary learning it should be stored in Problem.b1 matix
if isfield(Problem, 'b1')
    b = Problem.b1;
else
    b = Problem.b;            % The right-hand-side vector.
end
%%
if (solver.profile)
    fprintf('\nStarting solver %s... \n', solver.name);
end

start=cputime;
tStart=tic;
if strcmpi(solver.toolbox,'sparselab')
    y = eval([solver.name,'(SparseLab_A, b, n,',solver.param,');']);
elseif strcmpi(solver.toolbox,'sparsify')
    y = eval([solver.name,'(b, A, n, ''P_trans'', AT,',solver.param,');']);
elseif (strcmpi(solver.toolbox,'spgl1')||strcmpi(solver.toolbox,'gpsr'))
    y = eval([solver.name,'(b, A,',solver.param,');']);
elseif (strcmpi(solver.toolbox,'SPAMS'))
    y = eval([solver.name,'(b, A, solver.param);']);
elseif (strcmpi(solver.toolbox,'SMALL'))
    if isa(Problem.A,'float')
        y = eval([solver.name,'(A, b, n,',solver.param,');']);
    else
        y = eval([solver.name,'(A, b, n,',solver.param,',AT);']);
    end
elseif (strcmpi(solver.toolbox, 'ompbox'))
    G=A'*A;
    epsilon=solver.param.epsilon;
    maxatoms=solver.param.maxatoms;
    y = eval([solver.name,'(A, b, G,epsilon,''maxatoms'',maxatoms,''checkdict'',''off'');']);
elseif (strcmpi(solver.toolbox, 'ompsbox'))
    basedict = Problem.basedict;
     if issparse(Problem.A)
        A = Problem.A;
      else
        A = sparse(Problem.A);
      end
    G = dicttsep(basedict,A,dictsep(basedict,A,speye(size(A,2))));
    epsilon=solver.param.epsilon;
    maxatoms=solver.param.maxatoms;
    y = eval([solver.name,'(basedict, A, b, G,epsilon,''maxatoms'',maxatoms,''checkdict'',''off'');']);
    Problem.sparse=1;
%   To introduce new sparse representation algorithm put the files in
%   your Matlab path. Next, unique name <TolboxID> for your toolbox and
%   prefferd API <Preffered_API> needs to be defined.
%
% elseif strcmpi(solver.toolbox,'<ToolboxID>')
%
%     % - Evaluate the function (solver.name - defined in the main) with
%     %   parameters given above
%
%     y = eval([solver.name,'(<Preffered_API>);']);

else
    printf('\nToolbox has not been registered. Please change SMALL_solver file.\n');
    return
end

%%
%   Sparse representation time
tElapsed=toc(tStart);
solver.time = cputime - start;
if (solver.profile)
    fprintf('Solver %s finished task in %2f seconds (cpu time). \n', solver.name, solver.time);
    fprintf('Solver %s finished task in %2f seconds (tic-toc time). \n', solver.name, tElapsed);
end
solver.time=tElapsed;
% geting around out of memory problem when converting big matrix from
% sparse to full...

if isfield(Problem, 'sparse')&&(Problem.sparse==1)
    solver.solution = y;
else
    solver.solution = full(y);
end
if isfield(Problem,'reconstruct')
    %   Reconstruct the signal from the solution
    solver.reconstructed  = Problem.reconstruct(solver.solution);
end
end
