/* Part of DML (Digital Music Laboratory)
	Copyright 2014-2015 Samer Abdallah, University of London
	 
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

:- module(uripattern, [
		simplify/2
   ,  pattern_uri/2
	]).

:- use_module(library(semweb/rdf_db), [ rdf_global_id/2 ]).
:- use_module(library(uri), [ uri_encoded/3 ]).
:- use_module(library(dcg/basics)).

:- multifile uripattern:def/2.

%% pattern_uri(+P:uripattern,-X:uri) is det.
%% pattern_uri(-P:uripattern,+X:uri) is det.
pattern_uri(P,URI) :-
   (  var(URI) -> uri(P,Codes,[]), atom_codes(URI,Codes)
   ;  atomic(URI) -> atom_codes(URI,Codes), uri(P,Codes,[])
   ).

uri(P:A) --> !, {rdf_global_id(P:'',Q)}, atom(Q), uri(A).
uri(\(M)) --> !, {def(M,A)}, uri(A).
uri(num(N)) --> !, number(N).
uri(num(Len,N)) --> 
   { length(Codes,Len) }, 
   phrase(Codes),
   {  var(N) 
   -> number_codes(N,Codes)
   ;  format(codes(Codes),'~`0t~d~*+',[N,Len])
   }.
uri(tail(A)) --> !,
   (  {nonvar(A)} -> atom(A), eos
   ;  string_without("",C), eos, {atom_codes(A,C)}
   ).
uri(enc(A)) --> !,
   (  {nonvar(A)} -> {uri_encoded(path,A,B)}, atom(B) 
   ;  string_without("/",C), {atom_codes(B,C), uri_encoded(path,A,B)}
   ).
uri(A1/A2) --> !, uri(A1), "/", uri(A2).
uri(A)     --> {atomic(A)}, atom(A).

%% simplify(+P1:uri_pattern, -P2:uri_pattern) is nondet.
%  Non-deterministic maximal reduction of URI pattern.
simplify(E1,E3) :- 
   reduce(E1,E2) *-> simplify(E2,E3); E1=E3.

% simplify resource if possible, fails otherwise
% The pattern must be nonvar.
reduce(A,B) :- must_be(nonvar,A), reduce1(A,B).

reduce1(num(N),A)     :- number(N), atom_number(A,N).
reduce1(num(L,N),A)   :- number(N), format(atom(A),'~`0t~d~*+',[N,L]). % zero padded number
reduce1(\(A),B)       :- must_be(nonvar,A), def(A,B).
reduce1(enc(A),B)     :- atomic(A), uri_encoded(path,A,B).
reduce1(tail(A),A)    :- atomic(A).
reduce1(P:A,P:B)      :- reduce(A,B).
reduce1(P:A,PB)       :- atomic(A), rdf_global_id(P:A,PB).
reduce1(P:(A/B),PA/B) :- reduce(P:A,PA).
reduce1(A/B,A1/B)     :- reduce(A,A1).
reduce1(A/B,A/B1)     :- reduce(B,B1). % nb to get past these, A and B must be nonvar
reduce1(A/B,AB)       :- atom(A), atom(B), atomic_list_concat([A,B],'/',AB).
reduce1((A/B)/C,A/BC) :- reduce1(B/C,BC). 
reduce1(A/(B/C),AB/C) :- reduce1(A/B,AB).
reduce1((A/B)/C,A/(B/C)). 

