/* Part of DML (Digital Music Laboratory)
	Copyright 2014-2015 Samer Abdallah, University of London
	 
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

:- module(uripattern, [
		simplify/2
   ,  pattern_uri/2
	]).

:- use_module(library(semweb/rdf_db), [ rdf_global_id/2 ]).

:- multifile uripattern:def/2.

%% uri_constraint(+P:uripattern,?X:uri) is nondet.
%  Logical predicate implemented using delayed goals.
pattern_uri(A,B) :- must_be(nonvar,A), resource(A,B).

resource(P:A,PB) :- !, rdf_global_id(P:'',Q), plain(Q+A,PB).
resource(A,B) :- plain(A,B).

plain(\(M),B) :- def(M,A), (var(A) -> true; plain(A,B)).
plain(A1/A2,B)  :- plain(A1+'/'+A2,B).
plain(num(N),B) :- when((nonvar(N);nonvar(B)), atom_number1(B,N)).
plain(enc(A),B) :- when((nonvar(A);nonvar(B)), www_form_encode1(A,B)).
plain(atm(A),A).
plain(A1+A2,B) :- 
   freeze(A1,plain(A1,B1)),
   freeze(A2,plain(A2,B2)),
   when((ground(B1),ground(B2);ground(B)), atom_concat1(B1,B2,B)).
plain(A,A)     :- atom(A).

%% simplify(+P1:uri_pattern, -P2:uri_pattern) is nondet.
%  Non-deterministic maximal reduction of URI pattern.
simplify(E1,E3) :- 
   reduce(E1,E2) *-> simplify(E2,E3); E1=E3.

% simplify resource if possible, fails otherwise
% The pattern must be nonvar.
reduce(A,B) :- must_be(nonvar,A), reduce1(A,B).

reduce1(num(N),A)     :- number(N), atom_number(A,N).
reduce1(atm(A),A)     :- atomic(A).
reduce1(\(A),B)       :- nonvar(A), def(A,B).
reduce1(enc(A),B)     :- atomic(A), www_form_encode(A,B).
reduce1(P:A,Q+A)      :- rdf_global_id(P:'',Q).
reduce1(A/B,A+'/'+B).
reduce1(A+B,A1+B)     :- reduce(A,A1).
reduce1(A+B,A+B1)     :- reduce(B,B1). % nb to get past these, A and B must be nonvar
reduce1(A+B,AB)       :- atom(A), atom(B), atom_concat(A,B,AB).
reduce1((A+B)+C,A+BC) :- reduce1(B+C,BC). 
reduce1(A+(B+C),AB+C) :- reduce1(A+B,AB).

% ---------- support predicates ---------

atom_number1(Atom,Number) :-
   (nonvar(Number) -> number(Number); true),
   atom_number(Atom,Number).

www_form_encode1(Atom,Encoded) :-
   (nonvar(Encoded) -> atomic(Encoded); true),
   www_form_encode(Atom,Encoded).

atom_concat1(A,B,AB) :-
   (nonvar(AB) -> atomic(AB); true),
   atom_concat(A,B,AB).

