/* Part of DML (Digital Music Laboratory)
	Copyright 2014-2015 Samer Abdallah, University of London
	 
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

:- module(p2r_entailment,[ op(1200,xfx,<==), op(1200,xfx,==>), assert_all/1 ]).

/** <module> Prolog-to-RDF mapping entailment module

   This module provides a mechanism for exposing information in the Prolog
   database as RDF triples. The basis of this is a syntax for multiheaded
   Horn clauses with RDF triples in the head and Prolog goals in the body.
   The rules can be written in either direction: as =|Heads <== Body|=
   or =|Body ==> Heads|=, whichever is most convenient.

   There is also a grammar for writing  URI patterns. These are used to
   to build URIs from Prolog values and to parse URIs to extract Prolog
   values. The cost of this URI processing can be completely avoided by
   writing only variables in the head rdf/3 terms.
*/

:- use_module(library(uripattern_detdcg)).
:- use_module(library(semweb/rdf_db),[]).

:- rdf_meta rdf(r,r,o).
:- rdf_meta rdf(r,r,o,-).

:- multifile cliopatria:entailment/2.
:- multifile rdf/4.
:- op(1200,xfx,<==).
:- op(1200,xfx,==>).


cliopatria:entailment(p2r, p2r_entailment).

rdf(S,P,O) :- rdf(S,P,O,_).

user:term_expansion( (Body ==> Heads), Clauses) :-
   user:term_expansion( (Heads <== Body), Clauses).
user:term_expansion( (Heads <== Body), Clauses) :-
   debug(p2r,'Expanding: ~w',[Heads<==Body]),
	prolog_load_context(module,Mod),
   expand_clauses(Heads,Mod:Body,Clauses,[]).


expand_clauses((H1,H2),Body) --> !,
   expand_clauses(H1,Body),
   expand_clauses(H2,Body).

expand_clauses(rdf(SPat,PPat,OPat),Mod:Body) --> 
   {  call_dcg( ( opt_match(resource,SPat,S),
                  opt_match(resource,PPat,P),
                  opt_match(object,OPat,O) ), Mod:Body, Body1),
      Clause = (p2r_entailment:rdf(S,P,O,Mod) :- Body1), 
      debug(p2r,'Asserting clause: ~w',[Clause])
   },
   [Clause].

% builds an optimised goal to do pattern matching
opt_match(_,X,X) --> {var(X)}, !. 
opt_match(object,X,Y) --> {X=literal(_)}, !, {rdf_global_object(X,Y)}. 
opt_match(_,P1,URI) --> 
   {debug(p2r,'Simplifying URI pattern: ~w',[P1])},
   {simplify(P1,P2)}, opt_match(P2,URI).
opt_match(X,X) --> {atomic(X)}, !.
opt_match(P,U) --> by_inst(U,uripattern:pattern_uri(P,U)).

% this generates code to call G and G1 in an order determined
% by the instantiation state of X at run time.
by_inst(X,G1,G, (var(X) -> G, G1; G1, G)).


%% assert_all(+Module) is det.
%  This asserts all the RDF triples entailed by the p2r module Module.
%  Informational messages are printed, including the time taken.
assert_all(G) :-
   forall(p2r_entailment:rdf(S,P,O,G), rdf_assert(S,P,O,G)).

