/* Part of DML (Digital Music Laboratory)
	Copyright 2014-2015 Samer Abdallah, University of London
	 
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

:- module(echotools, [ echo/1, mb_to_echo/2, mb_to_spotify/2 ]).

:- use_module(echonest).
:- use_module(library(jpath)).
:- use_module(library(dictutils)).
:- use_module(library(insist)).
:- use_module(library(dcg_core)).
:- use_module(library(dcg_pair)).

echonest_key('NPIODUIKW9DJN1UCI').

echo(Path,Params,Decoder) :- 
   echonest_key(K), echocall(K,Path,Params,Resp),
   decode(response:Decoder,Resp).

decode(A,X) :- jpath(A,X).

echo(mood(X)) :- echo([artist,list_terms],[type=mood],terms:_#name:X).
echo(style(X)) :- echo([artist,list_terms],[type=style],terms:_#name:X).
echo(genre(X)) :- echo([genre,list],[],genres:_#name:X).
echo(genre_description(X,D)) :- var(X), !, echo_facets([genre,list],[],genres,[name:X,description:D]).
echo(genre_description(X,D)) :- must_be(text,X), !, echo_facets([genre,profile],[name=X],genres,[description:D]).
echo(search(genre,Term,X,D)) :- echo([genre,search],[name=Term,bucket=description],genres:_#(name:X,description(D))).

echo(genre_artist(Genre,Facets)) :- echo_facets([genre,artists],[name=Genre],artists,Facets).
echo(artist_genre(ArtistID,G)) :- echo([artist,profile],[id=ArtistID,bucket=genre],artist:genres:_#name:G).
echo(genre_facets(Facets)) :- echo_facets([genre,list],[],genres,Facets).
echo(songs(Ids,Facets,Opts)) :- 
   seqmap(param(id),Ids,Params,Opts),
   echo_facets([song,profile],Params,songs,Facets).

echo(search(song,Conditions,Facets)) :- 
   echo_facets([song,search],Conditions,songs,Facets).

echo(genre_song(Genre,Artist,Title,Id)) :-
   echo(genre_artist(Genre,[name:Artist,songs:_#(id:Id,title:Title)])).

echo(genre_songs_by_title(Genre,Artist,Title,Ids)) :-
   setof(Id,echo(genre_song(Genre,Artist,Title,Id)),Ids).

param(P,X) --> [P=X].

echo_facets(Path,Params0,Field,Facets) :- 
   included(Path,Incs),
   seqmap(facet(Incs),Facets,Params-Decs,Params0-[]),
   list_to_conj(Decs,Decoder),
   echo(Path,Params,\Response),
   catch( jpath(Field:_#Decoder,Response),
          Ex, ( print_message(warning,Ex), 
                format(user_error,"Response was:\n",[]),
                print_dict(user_error,Response,[]),
                fail)).

list_to_conj([D],D).
list_to_conj([D1,D2|Ds],(D1,D2s)) :- list_to_conj([D2|Ds],D2s).

facet(_,\X) --> !, \> [\X].
facet(Included,Name:Decoder) --> !,
   if(\+memberchk(Name,Included), \< [bucket=Name]),
   \> [Name:Decoder].

included(Path,Included) :- insist(included_db(Path,Included),unknown_includes(Path)).

included_db([genre,list],[name]).
included_db([genre,artists],[id,name]).
included_db([song,profile],[id,title,artist_id,artist_name]).
included_db([song,search],[id,title,artist_id,artist_name]).
included_db([genre,profile],[name]).

mb_to_echo(ID,Songs) :-
   atom_concat('musicbrainz:song:',ID,FID),
   findall(Song,
      echo(songs([FID],[\Song, tracks:1# \_], [ bucket='id:spotify'])),
      Songs).

%% mb_to_spotify(+MBId:atom,-SpURI:atom) is nondet.
%  Finds Spotify URIs corresponding to MusicBrainz recording IDs using
%  the Echonest Rosetta web service.
mb_to_spotify(ID,SpID) :-
   mb_to_echo(ID,Songs),
   member(Song,Songs),
   jpath(tracks:_#foreign_id:SpID,Song).

