/* Part of DML (Digital Music Laboratory)
	Copyright 2014-2015 Samer Abdallah, University of London
	 
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

:- module(echonest, 
   [	echocall/4
   ]).

/** <module> Interface to echonest Web API
 */

:- use_module(library(fileutils)).
:- use_module(library(http/json)).
:- use_module(library(jpath)).
:- use_module(library(webby)).
:- use_module(library(state)).

:- set_prolog_flag(double_quotes,string).

:- setting(min_wait,number,3,"Minimum time between echonest requests").
:- initialization set_state(next_request_time,0). 

wait_respectfully :-
   get_time(Now),
   setting(min_wait,TMin),
   get_state(next_request_time,T0), T1 is max(Now,T0) + TMin,
   set_state(next_request_time,T1),
   (  Now>=T0 -> true
   ;  DT is T0-Now, 
      debug(echonest,"Sleeping for ~f seconds to respect rate limit",[DT]),
      sleep(DT)
   ).

echocall(Key,PathParts,Params,Response) :-
   debug(echonest,"Echonest call: ~q, ~q...",[PathParts,Params]),
   parts_path([api,v4|PathParts],Path),
   wait_respectfully,
   catch( restcall(get([api_key=Key, format=json | Params]), json(Response), 
                   [protocol(http), host('developer.echonest.com'), path(Path)], []),
      Ex, handle(Ex)).

handle(http_bad_status(SC,codes(Doc))) :- !,
   with_input_from(codes(Doc),json_read_dict(current_input,Dict)),
   jpath(response:status:(code:Code,message:Msg),Dict),
   throw(echonest_error(SC,Code,Msg)).
handle(Ex) :- throw(Ex).   


prolog:message(echonest_error(SC,Code,Msg)) --> 
   {http_status(SC,Meaning)},
   ["(HTTP ~w: ~w) Echonest error ~w: '~s'."-[SC,Meaning,Code,Msg]].
