/* Part of DML (Digital Music Laboratory)
	Copyright 2014-2015 Samer Abdallah, University of London
	 
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

:- module(kern_analysis, []).

:- use_module(library(kerndata)).
:- use_module(library(grammars)).
:- use_module(library(compression)).
:- use_module(library(swipe)).

grammars:dataset_sequences(DS,Seqs) :-
   kerndata:dataset_sequences(DS,Seqs).



write_dataset_sequences(DS) :-
   dataset_sequences(DS,SS),
   with_stream_encoding(current_output,octet,
      maplist(write_seq(notenums),SS)).

write_dataset(Prep,DS) :-
   dataset_sequences(DS,SS),
   with_stream_encoding(current_output,octet,
      maplist(write_seq(Prep),SS)).

write_item(Prep,Item) :-
   item_sequence(Item,Sequence),
   write_seq(Prep,Sequence).

write_seq(notenums,NS) :-
   write_bytes(NS),
   write_bytes([255]).

write_seq(intervals,NS) :-
   nth1(1,NS,N1),
   nnums_ivals(NS,IS),
   append(IS2,[end],IS),
   maplist(plus(127),IS2,CS),
   write_bytes([N1|CS]),
   write_bytes([255]).

ds_complexity(Method,Prep,DS,K) :-
   complexity(Method,
      write_dataset(Prep,DS),
      K).

ds_conditional_complexity(Method,Prep,Ref,DS,K) :-
   conditional_complexity(Method,
      write_dataset(Prep,Ref),
      write_dataset(Prep,DS),
      K).

% ds_mutual_information(Method,Prep,DS1,DS2,I) :-
%    mutual_information(Method,
%       write_dataset(Prep,DS1),
%       write_dataset(Prep,DS2),
%       I).

arg_min(K-Meth,Goal,KMin-Best) :-
   aggregate_all(min(K,Meth),(Goal,Meth\=naive(_,_)),min(KMin,Best)).

:- use_module(library(lambda)).
:- use_module(library(mlserver)).
:- use_module(library(uri)).

exp(conditioning_scat(DS,Prep,Ref)) :-
   dataset_items(DS,Items),
   findall([KMin,CKMin], 
      (  member(I,Items), 
         min_complexity(_,write_item(Prep,I),KMin),
         min_conditional_complexity(_,write_dataset(Prep,Ref), 
                                      write_item(Prep,I),CKMin)
      ), XX), 
   ??scat(transpose(arr(XX))),
   ??xlabel(q(complexity)),
   ??ylabel(q('conditional complexity')).

exp(relative_erasure(DS,Prep)) :-
   dataset_items(DS,Items),
   findall(CKRel, 
      (  select(I,Items,Rest), 
         min_complexity(_,write_item(Prep,I),KMin),
         min_conditional_complexity(_,maplist(write_item(Prep),Rest), 
                                              write_item(Prep,I),CKMin),
         CKRel is 1- CKMin/KMin
      ), XX), 
   ??barh(XX),
   ??xlabel(q('1 - relative erasure complexity')),
   item_yticks(Items).

exp(odd_one_out(Prep,DS,DS2,N)) :-
   dataset_items(DS,Items),
   dataset_items(DS2,Items2),
   nth1(N,Items2,OddBall),
   AllItems=[OddBall|Items],
   findall(CKRel, 
      (  select(I,AllItems,Rest), 
         min_complexity(_,write_item(Prep,I),KMin),
         min_conditional_complexity(_,maplist(write_item(Prep),Rest), 
                                              write_item(Prep,I),CKMin),
         CKRel is 1 - CKMin/KMin
      ), XX), 
   ??barh(XX),
   ??xlabel(q('1 - relative erasure complexity')),
   item_yticks(AllItems).

exp(erasure(DS,Prep)) :-
   dataset_items(DS,Items),
   findall([KMin,CKMin], 
      (  select(I,Items,Rest), 
         min_complexity(_,write_item(Prep,I),KMin),
         min_conditional_complexity(_,maplist(write_item(Prep),Rest), 
                                      write_item(Prep,I),CKMin)
      ), XX), 
   ??barh(transpose(arr(XX))),
   ??legend(cell([q(isolated),q(erasure)])),
   ??xlabel(q(bits)),
   item_yticks(Items).

exp(compressibility(DS,Prep,Ref)) :-
   dataset_items(DS,Items),
   findall([CKMin,KMin], 
      (  member(I,Items), 
         % complexity(id,write_item(Prep,I),L), 
         min_complexity(_,write_item(Prep,I),KMin),
         min_conditional_complexity(_,write_dataset(Prep,Ref), 
                                      write_item(Prep,I),CKMin)
         % arg_min(K-C, conditional_complexity(naive(C),write_dataset(Prep,Ref),
         %                                              write_item(Prep,I),K), NCK-_)
      ), XX), 
   ??barh(transpose(arr(XX))),
   item_yticks(Items),
   ??xlabel(q(bits)),
   ??legend(cell([q('delta compressed'),q(compressed)])).

exp(info_density(DS,Prep,Ref)) :-
   dataset_items(DS,Items),
   findall([H,HRef], 
      (  member(I,Items), 
         item_sequence(I,Pitches),
         length(Pitches,L),
         min_complexity(_,write_item(Prep,I),KMin),
         min_conditional_complexity(_,write_dataset(Prep,Ref), 
                                      write_item(Prep,I),CKMin),
         H is KMin/L,
         HRef is CKMin/L
      ), XX), 
   ??barh(transpose(arr(XX))),
   ??legend(cell([q(isolated),q(conditional)])),
   ??xlabel(q('bits per note')),
   item_yticks(Items).

exp(ds_similarity_matrix(Prep,Meth,Datasets)) :-
   ncd_matrix(ncd(Meth), write_dataset(Prep), Datasets, Matrix),
   ncd_image(Matrix),
   yticks(Datasets).

exp(similarity_matrix(Prep,Meth,DS)) :-
   dataset_items(DS,Items),
   ncd_matrix(ncd(Meth), write_item(Prep), Items, Matrix),
   ncd_image(Matrix),
   item_yticks(Items).

exp(conditional_similarity_matrix(Prep,Method,DS,Ref)) :-
   dataset_items(DS,Items),
   ncd_matrix(conditional_ncd(Method,write_dataset(Prep,Ref)), write_item(Prep), Items, Matrix),
   ncd_image(Matrix),
   item_yticks(Items).

exp(ds_conditional_similarity_matrix(Prep,Method,Datasets,Ref)) :-
   ncd_matrix(conditional_ncd(Method,write_dataset(Prep,Ref)), write_dataset(Prep), Datasets, Matrix),
   ncd_image(Matrix),
   yticks(Datasets).

ncd_image(Matrix) :- ??imagesc(1-arr(Matrix)), ??colorbar.

ncd_matrix(Sim,Writer,Items,Matrix) :-
   length(Items,N),
   numlist(1,N,IX),
   maplist(ncd_matrix_column(Sim,Writer,Items,IX),IX,Matrix).

ncd_matrix_column(Sim,Writer,Items,JX,I,Column) :-
   maplist(ncd_matrix_entry(Sim,Writer,Items,I),JX,Column).

ncd_matrix_entry(Sim,Writer,Items,I,J,D) :-
   (  I=J -> D=nan
   ;  nth1(I,Items,XI),
      nth1(J,Items,XJ),
      call(Sim, call(Writer,XI), call(Writer,XJ), D)
   ).

item_yticks(Items) :-
   length(Items,NumItems),
   maplist(item_label,Items,Labels),
   ??yticks(1:NumItems,cell(Labels)).

item_label(Item,q(Name)) :-
   item_uri(Item,URI),
   uri_components(URI,Components),
   uri_data(path,Components,Path),
   directory_file_path(_,File,Path),
   file_name_extension(Name,_,File).

printfig(W,H,Name) :-
   format(atom(Path),'/Users/samer/pubs/asymmus/workshop/figs/~w.eps',[Name]),
   format(atom(Cmd),'epstopdf ~w',[Path]),
   ??fsetup(W,H,q(centimeters)),
   ??print(q('-depsc2'),q(Path)),
   shell(Cmd).

plot_method_rankings :-
   setof( r(MeanRank,Method,Ranks),
          ( compression:method_rankings(Method,Ranks),
            mean_list(Ranks,MeanRank)),
          Results),
   ??newplot,
   ??hold(q(on)),
   forall( nth1(I,Results,r(_,_,Ranks)),
           ??plot(Ranks,I+0.7*(rand(size(Ranks))-0.5),q('.'))),
   ??hold(q(off)),
   maplist(result_method,Results,Methods),
   yticks(Methods).

result_method(r(_,Method,_),Method).

yticks(Terms) :-
   length(Terms,N),
   maplist(term_label,Terms,Labels),
   ??yticks(1:N,cell(Labels)).

term_label(Term,q(Label)) :- format(atom(Label),'~w',[Term]).


mean_list(L,Mean) :-
   length(L,N),
   sumlist(L,Sum),
   Mean is Sum/N.
    

% pairwise(DS,Matrix) :-
%    dataset_items(DS,Items),
%    maplist(\I1^maplist(\I2^conditional_complexity(


% rnd_state(T) :- get_rnd_state(S), rnd_state_term(S,T).

% with_sample(T,RV,Pred) :-
%    rnd_state_term(S1,T),
%    sample(RV,X,S1,S2),
%    set_rnd_state(S2),
%    call(Pred,X).

