/* Part of DML (Digital Music Laboratory)
	Copyright 2014-2015 Samer Abdallah, University of London
	 
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

:- module(htmlutils, 
      [  link//2
      ,  style//1
      ,  script//1
      ,  use_font//2
      ,  use_font//3
      ,  paginator//3
      ,  element//2
      ]).

:- use_module(library(http/html_write)).
:- use_module(library(http/html_head)).
:- use_module(library(http/http_path)).
:- use_module(library(http/http_dispatch)).
:- use_module(library(http/http_server_files)).

:- multifile user:body//1, user:body//2.
:- multifile user:head//1, user:head//2.
% :- multifile user:style/2.

:- set_prolog_flag(double_quotes,string).

% :- setting(appname, string, "anApp", "Application name").

:- meta_predicate element(+,:,?,?).
element(Element,Content) --> { Item=..[Element, Content] }, html(Item).

link(Id,Text) --> html(a(href(location_by_id(Id)),Text)).
style(Loc) --> {http_absolute_location(Loc,Ref,[])}, html(link([rel(stylesheet), href(Ref)],[])).
script(Loc) --> {http_absolute_location(Loc,Ref,[])}, html(script(src(Ref),[])).

http:location(googlefonts,"//fonts.googleapis.com",[]).

user:term_expansion((:- googlefont(Name,Family)), Decls) :-
   Decls = [ (:- html_resource(Name,[virtual(true),requires(googlefonts(Query))]))
           , (:- html_resource(googlefonts(Query),[mime_type(text/css)]))
           ],
  string_concat("css?family=",Family,Query).


use_font(Family,Resource) -->
   html_requires(Resource),
   html_post(head, style(type="text/css","html,body,h1,h2,h3,h4,h5,h6,select,input,.btn {font-family: '~s';}"-[Family])).

use_font(Elements,Family,Resource) -->
   html_requires(Resource),
   html_post(head, style(type="text/css","~w {font-family: '~s';}"-[Elements,Family])).

paginator(_,_,1) --> [].
paginator(Handler,Page,Pages) -->
   html(div(class('text-centered'),
      ul(class(pagination),
           [ li(\prev(Page,Pages,Handler))
           , li(\page_link(Handler,1,1))
           , li(span("~d of ~d"-[Page,Pages]))
           , li(\page_link(Handler,Pages,Pages))
           , li(\next(Page,Pages,Handler))
           ]))).

prev(1,_,_) --> !,html(span(class(inactive),&(larr))).
prev(N,_,Handler) --> { succ(M,N) }, page_link(Handler,M,&(larr)).

next(N,N,_) --> !,html(span(class(inactive),&(rarr))).
next(N,_,Handler) --> { succ(N,M) }, page_link(Handler,M,&(rarr)).

page_link(ID-Params,N,Content) -->
   { http_link_to_id(ID,[page(N)|Params],Link) },
   html(a(href(Link),Content)).


