:- module(dml_c3,
   [  csv_op_chart/3
   ,  cla_op_chart/3
   ,  cla_label_lang/3
   ,  c3_bar/3, c3_bar/2
   ,  c3_plot/3
   ,  c3_steps/3
   ,  c3_csteps/3
   ,  c3_hist/5
   ]).

:- use_module(library(computations)).
:- use_module(library(mlserver)).
:- use_rendering(c3).

csv_op_chart(Op,Result,Chart) :- op_chart(Op,Result,Chart), !.
csv_op_chart(Op,X,unrecognised(Op,X)).

op_chart(pitch_hist(_Weighting),Pairs,Ch) :- 
    unzip(Pairs,NoteNums,Counts),
    maplist(computations:pitch_number_name,NoteNums,Pitches),
    c3_bar(pitch-Pitches,counts-Counts,Ch1),
    Ch=Ch1.put(axis/x/type,categorical).

op_chart(freq_hist(Map,_),Counts,Ch) :- c3_hist(bars,pitch,Map,Counts,Ch).
op_chart(freq_hist_r(Map,_),Counts,Ch) :- c3_hist(bars,pitch,Map,Counts,Ch).
op_chart(tempo,Pairs,Ch) :- unzip(Pairs,T,X), c3_plot(time-T,tempo-X,Ch).
op_chart(uniform_tempo(_),T-X,Ch) :- c3_plot(time-T,tempo-X,Ch).
op_chart(uniform_tempo_r(_),T-X,Ch) :- c3_plot(time-T,tempo-X,Ch).
op_chart(uniform_tempo(_,_),T-X,Ch) :- c3_plot(time-T,tempo-X,Ch).
op_chart(uniform_tempo_r(_,_),T-X,Ch) :- c3_plot(time-T,tempo-X,Ch).
op_chart(tempo_hist_r(_DT,Map),_-Counts,Ch) :- c3_hist(steps,tempo,Map,Counts,Ch).
op_chart(tempo_hist(_DT,Map),_-Counts,Ch) :- c3_hist(steps,tempo,Map,Counts,Ch).
op_chart(normalised_tempo(N),_T-X,Ch) :- tempo_curve(N,X,Ch).
op_chart(normalised_tempo_r(N),_T-X,Ch) :- tempo_curve(N,X,Ch).

cla_op_chart(Op,X,Chart) :- cla_chart(Op,X,Chart), !.
cla_op_chart(Op,X,unrecognised(Op,X)).

cla_chart(Op,X,Chart) :-
    is_dict(X), _{result:R} :< X, !,
    cla_chart(Op,R,Chart).

cla_chart(collection_pitch_histogram(_),X,Chart) :-
   c3_bar(notenum-X.values,counts-X.counts,Chart).    

cla_chart(collection_freq_histogram(Min,Max,Quant,_),X,Chart) :- % matlab
    c3_hist(bars,pitch,binmap(Min,Max,(Max-Min)*Quant+1),X.counts,Chart).

cla_chart(collection_freq_histogram(_Lang,Min,Max,Quant,_),X,Chart) :-
    c3_hist(bars,pitch,binmap(Min,Max,(Max-Min)*Quant+1),X.counts,Chart).

cla_chart(collection_tempo_histogram(_,Min,Max,N),X,Chart) :- % matlab
    c3_hist(steps,tempo,expmap(Min,Max,N),X.counts,Chart).

cla_chart(collection_tempo_histogram(_Lang,_,Min,Max,N),X,Chart) :-
    c3_hist(steps,tempo,expmap(Min,Max,N),X.counts,Chart).

cla_chart(collection_tempo_curve(N),X,Chart) :- % matlab
    tempo_curve(N,X.means,Chart).

cla_chart(collection_tempo_curve(_Lang,N),X,Chart) :-
    tempo_curve(N,X.means,Chart).

cla_chart(py_hist(tagged(tonic), key_tonic_hist:aggregate, []),X,Chart) :-
   c3_bar(key-X.values,counts-X.counts,Chart1),
   Chart=Chart1.put(axis/x/type,categorical).

cla_chart(py_hist(tagged(transcription), semitone_hist:aggregate, []),X,Chart) :-
   c3_bar('pitch class'-X.values,counts-X.counts,Chart1),
   Chart=Chart1.put(axis/x/type,categorical).

cla_chart(tuning_stats, X, Chart) :-
   Hist=X.stats.hist,
   c3_hist(steps,frequency,edgemap(Hist.edges),Hist.counts,Chart).

cla_chart(py_cla(tagged(transcription(1)), tuning_stats:per_file, []),X, Chart) :-
   cla_chart(tuning_stats, X, Chart).


tempo_curve(N,X,Chart) :- 
   numlist(1,N,T), 
   c3_csteps(time-T,tempo-X,Chart).

c3_hist(bars,XLabel,Map,Counts,Chart) :- 
    array_list(centres(Map),Centres1),
    maplist(fix(2),Centres1,Centres),
    c3_bar(XLabel-Centres,counts-Counts,Chart).

c3_hist(steps,XLabel,Map,Counts,Chart) :- 
    array_list(edges(Map),Edges1),
    maplist(fix(2),Edges1,Edges),
    append(Counts,[0],C),
    c3_steps(XLabel-Edges,counts-C,Chart).


fix(N,X,Y) :- Q is 10^N, Y is round(X*Q)/Q.

:- public c3:put//2
        , c3:axis//2
        , c3:bar//2
        , c3:area//2
        , c3:zoom//1
        , c3:legend//1
        , c3:subchart//1
        , c3:put//2
        , c3:scat//2
        .

c3:put(Path,Val,D1,D1.put(Path,Val)).
c3:axis(Name,Label,D1,D2) :- axis_pos(Name,Pos), D2=D1.put(axis/Name/label,_{text:Label,position:Pos}).
c3:axes(Lab1,Lab2) --> c3:axis(x,Lab1), c3:axis(y,Lab2).
c3:legend(F) --> c3:put(legend/show,F).
c3:subchart(F) --> c3:put(subchart/show,F).
c3:zoom(F) --> c3:put(zoom/enabled,F).

c3:scat(XLabel,YLabel) --> c3:init(scatter,XLabel,YLabel).
c3:bar(XLabel,YLabel) --> c3:init(bar,XLabel,YLabel).
c3:area(XLabel,YLabel) --> c3:init(area,XLabel,YLabel).

c3:init(Type,XLabel,YLabel) -->
   c3:axes(XLabel,YLabel),
   c3:put(data,_{columns:[],type:Type}),
   c3:put(axis/x/tick/fit,false).

axis_pos(x,'outer-center').
axis_pos(y,'outer-middle').

c3_plot(XL-XV,YL-YV, c3{data:_{x:XL,columns:[[XL|XV],[YL|YV]]}}.axes(XL,YL).legend(false)).
c3_steps(XL-XV,YL-YV, c3{ line:_{step:_{type:'step-after'}}, 
                          data:_{type:'area-step',x:XL, columns:[[XL|XV],[YL|YV]]}
                        }.axes(XL,YL).legend(false)).
c3_csteps(XL-XV,YL-YV, c3{data:_{type:step,x:XL,columns:[[XL|XV],[YL|YV]]}}.axes(XL,YL).legend(false)).
c3_bar(Label-Vals,c3{data:_{type:bar, columns:[[Label|Vals]]}}.axis(y,Label)).
c3_bar(XL-XV,YL-YV,c3{data:_{type:bar, x:XL, columns:[[XL|XV],[YL|YV]]}}.axes(XL,YL).legend(false)).


cla_label_lang(collection_pitch_histogram(_),pitch_histogram,pl).
cla_label_lang(collection_freq_histogram(_,_,_,_),freq_histogram,ml). 
cla_label_lang(collection_freq_histogram(Lang,_,_,_,_),freq_histogram,Lang).
cla_label_lang(collection_tempo_histogram(_,_,_,_),tempo_histogram,ml). % matlab
cla_label_lang(collection_tempo_histogram(Lang,_,_,_,_),tempo_histogram,Lang).
cla_label_lang(collection_tempo_curve(_),tempo_curve,ml). % matlab
cla_label_lang(collection_tempo_curve(Lang,_),tempo_curve,Lang).
cla_label_lang(py_hist(tagged(tonic), key_tonic_hist:aggregate, []),tonic_histogram,py).
cla_label_lang(py_hist(tagged(transcription), semitone_hist:aggregate, _),pc_histogram,py).
cla_label_lang(tuning_stats, tuning_stats, py).
cla_label_lang(py_cla(tagged(transcription(1)), tuning_stats:per_file, _),tuning_stats, py).
cla_label_lang(py_cla(keys_chords,chord_seq_key_relative:aggregate,_),key_relative_chord_seq,py).
cla_label_lang(py_cla(similarity_bundle,similarity:per_file,_), similarity, py).


