/* Part of DML (Digital Music Laboratory)
	Copyright 2014-2015 Samer Abdallah, University of London
	 
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

:- module(dml,
      [ call_debugging/2
      , call_nodebug/1
      , view_computations/0
      , start_matlab/0
      , stop_matlab/0
      , start_r_server/0
      , update/1 ]).

/** <module> DML main loader
 */

:- set_prolog_flag(double_quotes, string).
:- debug_message_context(+time('%FT%T')).

user:file_search_path(memo, cp_application(memo_db)).

:- use_module(cp_application('config-enabled/dml_permission')).

:- use_module(library(rcutils)).
:- use_module(library(http/http_dispatch)).
:- use_module(skin(minimal)).
:- use_module(library(dml_data)).
:- use_module(applications(callgraph_ui)).
:- use_module(applications(audio_ui)).
:- use_module(applications(score_ui)).
:- use_module(applications(csv_ui)).
:- use_module(applications(memo_ui)).
:- use_module(applications(transcription_ui)).
:- use_module(applications(dml_overview)).

:- use_module(library(code_cache)).
:- use_module(library(dml_crawler)).
:- use_module(library(dml_spotify)).
:- use_module(library(dml_musicbrainz)).
:- use_module(library(dml_swish)).
:- use_module(library(dml_misc)).
:- use_module(library(musiclab)).
:- use_module(library(rdfutils)).
:- use_module(library(dlogic)).
:- use_module(library(dataset)).
:- use_module(library(mlserver), except([start_matlab/0,start_matlab/1])).
:- use_module(library(real)).
:- use_module(library(backend_json)).
:- use_module(library(dovamp)).
:- use_module(api(matlab)).
:- use_module(api(r_plot)).
:- use_module(api(archive)).
:- use_module(api(dmlvis)).
:- use_module(api(perspectives)).
:- use_module(api(transcription)).
:- use_module(components(audio)).
:- use_module(cliopatria(hooks)).

cliopatria_openid:insecure_host('mirg.city.ac.uk').


% Hooks for audio related services
cp_audio:audio_file(URI,File,Fmt) :- beets_p2r:audio_file(URI,File,Fmt).
cp_audio:audio_file(URI,File,Fmt) :- charm_p2r:audio_file(URI,File,Fmt).
cp_audio:audio_file(URI,File,Fmt) :- bl_p2r:audio_file(URI,File,Fmt).
cp_audio:audio_file(URI,File,Fmt) :- mazurka_p2r:audio_file(URI,File,Fmt).

cp_audio:audio_link(URI,Link,just(mp3)) :- bl_p2r:audio_link(URI,Link).
cp_audio:audio_link(URI,URL,just(Fmt)) :-
   bl_p2r:audio_file(URI,File,just(Fmt)),
   file_permission(File,public),
   http_link_to_id(audio_get,[uri(URI),format(Fmt)],URL).
cp_audio:audio_link(URI,Link,just(mp3)) :- charm_p2r:audio_link(mp3,URI,Link).
cp_audio:audio_link(URI,URL,just(Fmt)) :-
   Formats=[ogg,mp3], % in order of preference
   (  beets_p2r:audio_file(URI,File,just(Orig))
   ;  mazurka_p2r:audio_file(URI,File,just(Orig))
   ),
   file_permission(File,public),
   (member(Orig,Formats) -> member(Fmt,Formats); member(Fmt,[Orig|Formats])),
   http_link_to_id(audio_get,[uri(URI),format(Fmt)],URL).

%:- rdf_set_cache_options([enabled(true),global_directory(cache)]).

% ------------------ string portrayal -------------
% user:portray(String) :-
%    string(String), !,
%    truncate(50,String,Short),
%    writeq(Short).

% truncate(Max,S,S) :- string_length(S,L), L<Max, !.
% truncate(Max,S1,S3) :-
%    L is Max-3,
%    sub_string(S1,0,L,_,S2),
%    string_concat(S2,"...",S3).

% Adjust menus
cliopatria:menu_item(900=query/isearch, 'Interactive faceted search').
cliopatria:menu_item(900=places/http_skos_browser, 'SKOS Concept browser').
cliopatria:menu_item(800=help/dml_overview, 'DML overview').
cliopatria:menu_item(800=admin/job_view, 'Current jobs').

warm_indices :-
    ignore(rdf(s, _, _)),
    ignore(rdf(_, p, _)),
    ignore(rdf(_, _, o)),
    ignore(rdf(s, p, _)),
    ignore(rdf(_, p, o)),
    ignore(rdf(s, p, o)),
    ignore(rdf(_, _, _, g)),
    ignore(rdf(s, _, _, g)),
    ignore(rdf(_, p, _, g)).

% :- rdf_register_prefix(dmlcla,'http://dml.org/cla#').
% :- initialization rdf_load(dml('cla/ontologies/dmlclaOntology.n3')).
:- initialization open_log, maplist(debug,[dmlvis(perspective),dmlvis(error),dmlvis(response)]).
:- initialization persistent_history('.dml.history',[interval(300)]), confirm_on_halt.
:- initialization p2r_import(memo_p2r).
:- initialization 
      (  current_periodical(_,clean_code_cache(_),Id)
      -> remove_periodical(Id)
      ;  true
      ),
      periodically(3600,clean_code_cache(3600)).

:- dynamic log_stream/1.
open_log :-
   (log_stream(S) -> close(S), retractall(log_stream(_)); true),
   expand_file_name('~/var/log/dml.log',[LogFile]),
   open(LogFile,append,LogStream,[buffer(line)]), nl(LogStream),
   assert(log_stream(LogStream)),
   debug(log), debug(log,'Started logging.',[]).


:- meta_predicate call_debugging(?,0), call_nodebug(0).
call_debugging(Topic,Goal) :- setup_call_cleanup(debug(Topic), Goal, nodebug(Topic)).
call_nodebug(Goal) :-
   setup_call_cleanup( (setof(T,debugging(T),Topics),maplist(nodebug,Topics)), Goal, maplist(debug,Topics)).

view_computations :-
   call_debugging(computations(item),termutils:with_status_line(get_single_char(_))).

logging(dmlvis(method)).
logging(dmlvis(perspective)).
logging(log).

prolog:debug_print_hook(computations(item),Fmt,Args) :-
   termutils:status(Fmt,Args).

prolog:debug_print_hook(Topic,Fmt,Args) :-
   logging(Topic), log_stream(S), 
   get_time(T),
   format(S,'~@ ~w | ~@\n',[format_time(current_output,'%FT%T',T), Topic, format(Fmt,Args)]),
   fail.

update(dml) :-
   debug(log,'Updating dml-cliopatria repository.',[]),
   shell('cd cpack/dml && hg pull && hg update'),
   make.

start_matlab :- start_matlab([]).
start_matlab(Opts) :- mlserver:start_matlab([path([dml(matlab)])|Opts]).
start_r_server :- real:r_start_server.

:- multifile sandbox:safe_primitive/1.
:- multifile sandbox:safe_directive/1.
sandbox:safe_primitive(dml_permissions:file_permission(_,_)).
sandbox:safe_primitive(cp_audio:audio_link(_,_,_)).

initiate_bl_scraping(Status) :-
   format('% Initiating BL audio link scraping...\n',[]),
   async:async_memo(vis_cla, bl_p2r:scrape_all([],_), Status, [recompute(failed),status_var(time-progress(summary))]).

after_load :- 
   warm_indices,
   initiate_bl_scraping(_),
   start_r_server, 
   start_matlab. 
