/* Part of DML (Digital Music Laboratory)
	Copyright 2014-2015 Samer Abdallah, University of London
	 
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

:- module(cp_r_fig,
      [ figure//4
      , figure//3
      , term_rendering//3
      ]).

:- meta_predicate figure(0,+,+,+,?,?).
:- meta_predicate figure(0,+,+,?,?).

:- use_module(library(http/http_dispatch),[http_link_to_id/3]).
:- use_module(library(http/html_write)).
:- use_module(library(http/html_head)).
:- use_module(library(dcg_core)).
:- use_module(library(listutils)).
:- use_module(library(optutils)).
:- use_module(library(pengines_io)).
:- use_module(library(swish/render)).
:- use_module(library(code_cache)).
:- use_module(library(insist)).
:- use_module(library(real)).
:- use_module(components(smartimg)).

:- register_renderer(rfig,"Render R figures").

term_rendering(rfig(Code),_,Opts) --> 
   {rendering_options(Opts,Opts1)},
   figure(r(Code),Opts1).
term_rendering(rfig(Code,W,H),_,Opts) --> 
   term_rendering(rfig(Code),_,[width(W),height(H)|Opts]).
term_rendering(rfig(Code,FOpts),_,Opts) -->
   {merge_options(FOpts,Opts,Opts1)},
   term_rendering(rfig(Code),_,Opts1).

rendering_options -->
   seqmap(option_default_select,[module(_),numbervars(_),quoted(_)],_).


%% figure(+Code:callable, +Width:natural, +Height:natural, +Params:list)// is det.
%% figure(+Code:callable, +Width:natural, +Height:natural)// is det.
%
%  Component to render an R figure into a HTML code. Code must be a Prolog
%  goal that renders the desired figure.
%  The default figure format (eg SVG or PNG) is taken from the default_figure_format
%  setting. See figure_render/1 for valid parameters. 
figure(Code,W,H) --> figure(Code,[width(W),height(H)]).
figure(Code,W,H,Params) --> figure(Code,[width(W),height(H)|Params]).
figure(Code,Params) -->
   {  debug(r_fig,"figure(~q,~q)",[Code,Params]),
      process_options(Download,Smart,Code,Params,Params1),
      insist(option(width(W),Params1), missing_parameter(width)),
      insist(option(height(H),Params1), missing_parameter(height)),
      http_link_to_id(r_figure_render, Params1, URL)
   },
   (  {Download=true}
   -> { option_default_select(format(_),_,Params1,Params2)},
      html(div([\image(Smart,URL,W,H),br([]),'Download as:', \download(pdf,Params2), \download(eps,Params2)]))
   ;  image(Smart,URL,W,H)
   ).

download(F,Params) -->
   {http_link_to_id(r_figure_render,[format(F)|Params],URL)},
   html([' ',a([href(URL),download],F)]).


process_options(Download,Smart,Code) -->
   {setting(r_plot:default_figure_format,Fmt0)},
   seqmap(option_default_select, [format(Fmt), cache(Cache), downloadable(Download), smart(Smart)],
                                 [Fmt0,false,false,false]),
   (select_option(color_map(CM)) -> {term_to_atom(CM,CMA)}, cons(color_map(CMA)); []),
   (select_option(size(W,H)) -> cons(width(W)), cons(height(H)); []),
   {Cache=true -> cache_code(Download,Code,Code1); Code=Code1},
   {term_to_atom(Code1,CodeAtom)},
   cons(format(Fmt)),
   cons(code(CodeAtom)).
