/* Part of DML (Digital Music Laboratory)
	Copyright 2014-2015 Samer Abdallah, University of London
	 
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

:- module(csv_ui, []).

/** <module> UI for viewing CSV files
*/
:- use_module(library(http/html_write)).
:- use_module(library(http/html_head)).
:- use_module(library(http/http_dispatch)).
:- use_module(library(http/http_parameters)).
:- use_module(library(decoration)).
:- use_module(library(htmlutils), [paginator//3]).
:- use_module(library(csvutils),  [uri_to_csv/2]).
:- use_module(library(listutils), [drop/3,take/3]).
:- use_module(components(table),  [table_from_goal//2]).
:- use_module(components(icons)).
:- use_module(cliopatria(hooks)).

:- set_prolog_flag(double_quotes,string).

:- http_handler(root(dml/csv/view), csv_view, []).

decoration:resource_view(URI,_) -->
   { sub_string(URI,_,_,0,".csv") }, 
   { http_link_to_id(archive_get,[uri(URI)],EntryURL) },
   { http_link_to_id(csv_view,[uri(URI)],ViewURL) },
   html_requires(font_awesome),
   html([ a(href=EntryURL, [\icon(download)," Download"]), &('MediumSpace')
        , a(href=ViewURL, [\icon(table)," View"])]).

decoration:resource_decoration(URI,Link) -->
   { sub_string(URI,_,_,0,".csv") }, !, 
   { http_link_to_id(csv_view,[uri(URI)],ViewURL) },
   html_requires(font_awesome),
   html( span( [ a(href(ViewURL),\icon(table))
               , &(nbsp), \Link
               ])).

csv_view(Request) :-
   http_parameters(Request,
      [  uri(URI,     [ optional(false), description("URI of CSV file")]) 
      ,  page(Page,   [ nonneg, default(1) ]) 
      ,  limit(Limit, [ nonneg, default(50) ])
      ]),
   uri_to_csv(URI,Rows),
   length(Rows,Total),
   Offset is Limit*(Page-1),
   Pages is ceil(Total/Limit),
   insist(Page=<Pages),
   drop(Offset,Rows,Rows1),
   insist(Page=Pages -> Rows1=Rows2; take(Limit,Rows1,Rows2)),
   format(string(FullTitle),"CSV view for ~w",[URI]),
   reply_html_page(cliopatria(demo), [title(FullTitle)],
                   [ h1(FullTitle)
                   , \paginator(csv_view-[uri(URI),limit(Limit)],Page,Pages)
                   , \table_from_goal(csv_row(Rows2),[])
                   , \paginator(csv_view-[uri(URI),limit(Limit)],Page,Pages)
                   ]).

csv_row(Rows,Row) :- member(R,Rows), R=..[_|Row].

