/* Part of DML (Digital Music Laboratory)
	Copyright 2014-2015 Samer Abdallah, University of London
	 
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

:- module(uripattern, [
		object/2
	,	resource/2
	,	literal/2
	,	max_reduce/2
	]).

:- use_module(library(semweb/rdf_db), [ rdf_current_ns/2 ]).

:- multifile uripattern:macro/2.

object(A,B)    :- freeze(A, object1(A,B)).
literal(A,B)   :- freeze(A, literal1(A,B)).
resource(A,B)  :- freeze(A, resource1(A,B)).
plain(A,B)     :- freeze(A, plain1(A,B)).

object1(literal(A),literal(B)) :- !, literal(A,B).
object1(A,B) :- resource1(A,B).

literal1(type(T,A),type(T1,A1)) :- !, resource(T,T1), plain(A,A1).
literal1(A,B) :- plain1(A,B).

resource1(P:A,PB) :- !, rdf_current_ns(P,Q), atom_concat1(Q,B,PB), plain(A,B).  
resource1(A,B) :- plain1(A,B).

plain1(pat(L),B) :- freeze(L,conc(L,B)). 
plain1(num(N),B) :- when(nonvar(N);nonvar(B),atom_number1(B,N)).
plain1(mac(M),B) :- when(nonvar(M);nonvar(B),macro(M,B)).
plain1(enc(A),B) :- plain(A,C), when(nonvar(C);nonvar(B),www_form_encode(C,B)).
plain1(atm(A),A).
plain1(A1/A2,B) :- freeze(B,sub_atom(B,_,_,_,'/')), plain1(A1+'/'+A2,B). 
plain1(A1+A2,B) :- atom_concat1(B1,B2,B), plain(A1,B1), plain(A2,B2).
plain1(A,B)     :- (atom(A);atom(B)), A=B.

conc([],'').
conc([H|T],B) :- atom_concat1(H,TA,B), freeze(T,conc(T,TA)).

atom_number1(A,N) :- 
	catch(atom_number(A,N), error(syntax_error(_),_), fail).

% like atom_concat but with more modes, using frozen constraints.
atom_concat1(A,B,C) :-
	freeze(A, (A=''->B=C;true)),
	freeze(B, (B=''->A=C;true)),
	when( (	nonvar(A),nonvar(B)
			;	nonvar(B),nonvar(C)
			;	nonvar(A),nonvar(C)
			), atom_concat(A,B,C)).


% non-deterministic maximal reduction
max_reduce(E1,E3) :- 
   reduce(E1,E2) *-> max_reduce(E2,E3); E1=E3.

% simplify resource or literal expression if possible
reduce(A,A) :- var(A), !, fail.
reduce(A/B,A+'/'+B).
reduce(A/B,A1/B) :- reduce(A,A1).
reduce(A/B,A/B1) :- reduce(B,B1).
reduce(A/B/C,A/(B/C)).
reduce(A+B,A1+B)     :- reduce(A,A1).
reduce(A+B,A+B1)     :- reduce(B,B1).
reduce(A+B,AB)       :- atom(A), atom(B), atom_concat(A,B,AB).
reduce(A+B,_) :- (var(A); var(B)), !, fail.
reduce(A+B/C,(A+B)/C).
reduce((A+B)+C,A+BC) :- atom(B), atom(C), atom_concat(B,C,BC).
reduce(A+(B+C),AB+C) :- atom(A), atom(B), atom_concat(A,B,AB).
reduce(P:A,Q+A)      :- must_be(atom,P), 
                        (  rdf_current_prefix(P,Q) -> true
                        ;  throw(unknown_prefix(P))
                        ).

reduce(num(N),A)     :- number(N), atom_number(A,N).
reduce(atm(A),A)     :- atom(A).
reduce(mac(A),B)     :- atom(A), macro(A,B).
reduce(enc(A),B)     :- atom(A), www_form_encode(A,B).
reduce(enc(A),enc(B)):- reduce(A,B).
reduce(A+pat(P),pat([A|P])) :- atom(A), nonvar(P).
reduce(pat(P)+A,pat(P1))    :- atom(A), nonvar(P), concat(P,[A],P1).
reduce(pat([A1,A2|AX]),pat([A12|AX])) :- 
	atom(A1), atom(A2), atom_concat(A1,A2,A12).

reduce(literal(E),literal(E1)) :- reduce(E,E1).
reduce(type(T,E),type(T1,E))   :- reduce(T,T1).
reduce(type(T,E),type(T,E1))   :- reduce(E,E1).
reduce(lang(V,L),lang(V1,L)) :- reduce(V,V1).
reduce(lang(V,L),lang(V,L1)) :- reduce(L,L1).

