# Part of DML (Digital Music Laboratory)
# Copyright 2014-2015 Steven Hargreaves; Samer Abdallah, University of London
 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

# -*- coding: utf-8 -*-
__author__="hargreavess, abdallahs"

import sys
from csvutils import *
from pitchutils import *
from aggregate import *
from rdflib   import RDF, RDFS
from rdfutils import parse_xsd_duration, event_ns, tl_ns, af_ns
from n3Parser import get_rdf_graph_from_n3

# map from pitch names to pitch class numbers in 0..11
pitch_map = { pitch_name(i,oct):i for oct in range(0,9) for i in range(0,12) }

# data conversions to a list of (pitch_name:string,duration:float) pairs representation

def transcription_from_csv(filename):
    # we assume format time, duration, pitch, velocity, note_name 
    return csv_map_rows(filename,5,lambda row:(row[4],float(row[1])))

def transcription_from_n3(filename):
    graph=get_rdf_graph_from_n3(filename)
    notes = [ ( graph.value(ev, RDFS.label),
                parse_xsd_duration(graph.value(graph.value(ev,event_ns.time), tl_ns.duration)) )
              for ev in subject((RDF.type, af_ns.Note)) ]

def notes_histogram(notes):
    hist = 12*[0]
    for note in notes: hist[pitch_map[note[0]]] += note[1]
    return hist

# Compute aggregate pitch histogram from a list of input transcriptions.
def aggregate(transcriptions):
    parser_table = { 'n3':transcription_from_n3, 
                     'csv':transcription_from_csv }

    hist = 12*[0] # will be aggragate histogram
    def accum(f):
        h = notes_histogram(decode_tagged(parser_table,f))
        total = sum(h)
        for x in range(0, 12): hist[x] += h[x]/total
    stats=for_each(transcriptions,accum)
    return { 'result': discrete_hist(pitch_class_names,hist), 'stats':stats }
