/* Part of DML (Digital Music Laboratory)
	Copyright 2014-2015 Samer Abdallah, University of London
	 
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

:- module(uripattern, [
		simplify/2
   ,  pattern_uri/2
	]).

:- use_module(library(semweb/rdf_db), [ rdf_global_id/2 ]).

:- multifile uripattern:def/2.

%% pattern_uri(+P:uripattern,-X:uri) is det.
%% pattern_uri(-P:uripattern,+X:uri) is det.
pattern_uri(P,URI) :-
   (  var(URI)    -> uri(P,String), atom_string(URI,String)
   ;  atomic(URI) -> atom_string(URI,String), uri(P,String)
   ).

uri(P:A,QB)   :- 
   rdf_global_id(P:'',Q),
   (  var(QB) -> uri(A,B), string_concat(Q,B,QB)
   ;  string_concat(Q,B,QB), uri(A,B) 
   ).
uri(\(M),B) :- def(M,A), uri(A,B).
uri(num(N),B) :- number_string(B,N).
uri(enc(A),B) :- ( var(B) -> www_form_encode(A,B1), atom_string(B1,B); www_form_encode(A,B)).
uri(atm(A),B) :- atom_string(A,B).
uri(A1/A2,B)  :- uri(A1+'/'+A2,B).
uri(A1+A2,B)  :- 
   (  var(B) -> uri(A1,B1), uri(A2,B2), string_concat(B1,B2,B)
   ;  string_concat(B1,B2,B), uri(A1,B1), uri(A2,B2)
   ).
uri(A,B)      :- atomic(A), atom_string(A,B).


%% simplify(+P1:uri_pattern, -P2:uri_pattern) is nondet.
%  Non-deterministic maximal reduction of URI pattern.
simplify(E1,E3) :- 
   reduce(E1,E2) *-> simplify(E2,E3); E1=E3.

% simplify resource if possible, fails otherwise
% The pattern must be nonvar.
reduce(A,B) :- must_be(nonvar,A), reduce1(A,B).

reduce1(num(N),A)     :- number(N), atom_number(A,N).
reduce1(atm(A),A)     :- atomic(A).
reduce1(\(A),B)       :- nonvar(A), def(A,B).
reduce1(enc(A),B)     :- atomic(A), www_form_encode(A,B).
reduce1(P:A,Q+A)      :- rdf_global_id(P:'',Q).
reduce1(A/B,A+'/'+B).
reduce1(A+B,A1+B)     :- reduce(A,A1).
reduce1(A+B,A+B1)     :- reduce(B,B1). % nb to get past these, A and B must be nonvar
reduce1(A+B,AB)       :- atom(A), atom(B), atom_concat(A,B,AB).
reduce1((A+B)+C,A+BC) :- reduce1(B+C,BC). 
reduce1(A+(B+C),AB+C) :- reduce1(A+B,AB).

