classdef MTTMixedFeatureGenreBasicSm < MTTAudioFeature & handle
    % ---
    %
    % The usual worklow for these features constist of three steps
    % 1. extract: extracts the basic single-file dependent features
    % 2. define_global_transform: calculates the global feature
    %       transformation parameters
    % 3. finalise: applies the common transformations to a specific feature
    % ---
    
    properties(Constant = true)
        
        % svn hook
        my_revision = str2double(substr('$Rev$',  5, -1));
    end
       
    properties
        % ---
        % Set default parameters
        % ---
        my_params = MTTAudioFeature.inherited_params(...
            'MTTAudioFeatureBasicSm', ...
            ... % ---
            ... % following are GenreBasic parameters
            ... % --- 
            'pct_genres', 1 ... % 1/100 percentile genre tags used
        );
    end
    
    % ---
    % member functions
    % ---
    methods
        
        % ---
        % constructor: pointer to feature in database
        % ---
        function feature = MTTMixedFeatureGenreBasicSm(varargin)

            feature = feature@MTTAudioFeature(varargin{:});
        end

        % ---
        % extract feature data by combining genre adn basicsm features
        % ---
        function data = extract(feature, clip)
            
            % ---
            % get Basic Summary audio features. this includes possible
            % local normalisations
            % ---
            basicsm = clip.audio_features_basicsm(feature.my_params);
            
            % ---
            % get genre tag features
            % ---
            
            genrebasic = clip.genre_features_basic(feature.my_params);
            
            % save to features data field
            data.audio = basicsm;
            data.tags = genrebasic;
            
            % prepare field for final features
            data.final.vector = [];
            data.final.vector_info = struct(); 
            data.final.dim = 0;
           
            % save info data
            data.info.type = 'MTTMixedFeatureGenreBasicSm';
            data.info.owner_id = clip.id;
            data.info.creatorrev = feature.my_revision;
            
            % save parameters
            data.info.params = feature.my_params;
        end
        
        function define_global_transform(features)
        % calculate and set normalization factors from the group of 
        % input features. These features will be set for the full database
        
            if numel(features) == 1
                error ('Insert feature array for this method');
            end
            
            % ---
            % We collect all the relevant basicsm 
            % features and get the transform on this basis.
            % ---
            for i = 1:numel(features)
            	basicsm(i) = features(i).data.audio;
            end
            
            % call the features own transsform function
            basicsm.define_global_transform();
            
            % ---
            % We collect all the relevant genretag 
            % features and get the transform on this basis.
            % ---
            for i = 1:numel(features)
            	genrebasic(i) = features(i).data.tags;
            end
            
            % call the features own transsform function
            genrebasic.define_global_transform();
            
            % ---
            % set common feature values for mixed features
            % ---
            features(1).my_db.set_common([1]); %trivial common
        end
        
        
        function finalise(feature)
        % applies a final transformation and collects the 
        % information of this feature within a single vector
        % see info for types in specific dimensions

            for i = 1:numel(feature)
                
                % check for neccesary parameters
                if isempty(feature(i).my_db.commondb)

                    error('Define the global transformation first');
                end
                
                % ---
                % finalise audio feature and get vector
                % ---
                basicsm = feature(i).data.audio;
                basicsm.finalise();
                
                % finalise tag features
                genrebasic = feature(i).data.tags;
                genrebasic.finalise;
                
                % ---
                % final data assembly
                % ---
                
                % concatenate vectors
                feature(i).data.final.vector = ...
                    [basicsm.vector() ; genrebasic.vector()];
                
                % add feature dimensions
                feature(i).data.final.dim = basicsm.dim + genrebasic.dim;
                
                % concatenate labels
                feature(i).data.final.vector_info.labels = ...
                    {basicsm.data.final.vector_info.labels{:}, ...
                    genrebasic.data.final.vector_info.labels{:}};
            end
        end
        
        % ---
        % destructor: do we really want to remove this 
        % from the database? No, but 
        % TODO: create marker for unused objects in db, and a cleanup
        %  function
        % ---
        function delete(feature)
            
        end
       
        function visualise(feature)
        % ---
        % plots the different data types collected in this feature
        % ---
            for i = 1:numel(feature)
                clip = MTTClip(feature(i).owner_id());
            end
        end
    end
end