#!/bin/sh

# shell script for automatic uploading/compiling of pd patch onto bbb
# Christian Heinrichs 2015
#

trap "{ echo "";exit 0; }" 2 

workingdir=".."
pdpath=""
NO_UPLOAD="0"
WATCH="0"
FORCE="0"
#make sure the paths have the trailing / . 
projectpath="../tmp/heavy/hvtemp/"
BBB_DEFAULT_PROJECT_NAME="heavyProject"
COMMAND_ARGS=
RUN_PROJECT=1
RUN_IN_FOREGROUND=0
RUN_WITHOUT_SCREEN=1
BELA_PYTHON27=

SCRIPTDIR=$(dirname "$0")
[ -z $SCRIPTDIR ] && SCRIPTDIR="./" || SCRIPTDIR=$SCRIPTDIR/ 
[ -z $HVRESOURCES_DIR ] && HVRESOURCES_DIR=$SCRIPTDIR/hvresources/
. $SCRIPTDIR.bela_common || { echo "You must be in Bela/scripts to run these scripts" | exit 1; }  

if [ -z "$BELA_PYTHON27" ]; then
    for PY in python python2.7 ; do
        python --version 2>&1 | grep "2\.7" >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            BELA_PYTHON27=$PY
            break;
        fi;
    done;
fi;

if [ -z "$BELA_PYTHON27" ]; then
    echo "It looks like you might not have python2.7 installed. If you do, please specify the path
    to your python2.7 executable in the environmental variable \$BELA_PYTHON27"
    exit 1;
fi;


usage ()
{
printf "\nUSAGE: build_pd.sh [[-i input folder containing _main.pd file ]\
 [-o output folder for temp heavy project .c files (default $projectpath)]\
 [-b remote path to copy to (default ~/Bela)] | [-h] | [-w|--watch] | [-n|--noupload] | [-r|--release arg]\n"
printf "\nexample: build_pd.sh -i ../projects/heavy/pd/hello-world -o ../projects/heavy/hello-world\n"
echo "If --watch is selected, the script will check every 1s for any file that is modified in the source folder, which re-triggers\
the building process.
If --screen is selected, the prompt returns to the user after launching Bela in a screen on the target device.
If --screen and --watch are combined, while the process is running in the screen, modifications to the source files will \
still trigger a new build.
-r allows to build against a specific Heavy release. Default is the most recent version.
"
}

while [ "$1" != "" ]; do
    case $1 in
        -b | --belaPath )       shift
                                BBB_BELA_HOME=$1
                                ;;
        -i | --input )          shift
                                pdpath=$1
                                ;;
        -o | --output )         shift
                                projectpath=$1
                                ;;
        -r | --release )        shift
                                release=$1
                                ;;
        -s | --screen )         RUN_WITHOUT_SCREEN="0"
                                ;;
        -w | --watch )          WATCH=1
                                ;;
        -n | --noupload )       NO_UPLOAD=1
                                ;;
        -h | --help )           usage
                                exit
                                ;;
        * )                     usage
                                exit 1
    esac
    shift
done

[ -z "$ENZIENAUDIO_COM_PATCH_NAME" ] && ENZIENAUDIO_COM_PATCH_NAME=bela
[ "$NO_UPLOAD" -eq 0 ] && [ -z "$pdpath" ] && { echo "Error: a path to the source folder should be provided"; exit 1; }

if [ -z "$release" ]
then 
  RELEASE_STRING=
else 
  RELEASE_STRING="-r $release"
fi

#create destination folder if it does not exist"
mkdir -p "$projectpath"

# These files will be cleared from $projectpath before calling uploader.py
#TODO: get a reliable, exhaustive, up-to-date list.
HEAVY_FILES='Heavy* Hv*'

set_date
reference_time_file="$projectpath"/

uploadBuildRun(){
    if [ $NO_UPLOAD -eq 0 ]; then
        # remove old static files to avoid obsolete errors
	# make sure the path is not empty, so avoiding to rm -rf / by mistake 
	[ -z $projectpath ] && { echo 'ERROR: $projectpath is empty.'; exit 0; } 
        # use -rf to prevent warnings in case they do not exist
        for file in $HEAVY_FILES
	do 
	    rm -rf "$projectpath"/$file
	done

        # invoke the online compiler
        "$BELA_PYTHON27" $HVRESOURCES_DIR/uploader.py "$pdpath"/ -n $ENZIENAUDIO_COM_PATCH_NAME -g c -o "$projectpath" $RELEASE_STRING ||\
            { echo "ERROR: an error occurred while executing the uploader.py script"; exit 1; }
    fi;

    echo "";

    # Test that files have been retrieved from the online compiler.
    # TODO: find a more reliable way of doing this. e.g.: have uploader.py fail with a non-zero error code.
    for file in $HEAVY_FILES;
    do
        ls "$projectpath"/$file >/dev/null 2>&1 || { printf "The online compiler did not return all the files or failed without notice, please try again and/or change HEAVY_FILES to be less strict.\n\n"; exit 1; }
    done

    # Apply any Bela-specific patches here 
    cp "$HVRESOURCES_DIR/HvUtils.h" $projectpath/ || exit 1;

    BBB_PROJECT_FOLDER=$BBB_PROJECT_HOME"/"$BBB_PROJECT_NAME #make sure there is no trailing slash here
    BBB_NETWORK_TARGET_FOLDER=$BBB_ADDRESS:$BBB_PROJECT_FOLDER

    # check how to copy/sync render.cpp file...
    # check if custom heavy/render.cpp file is provided in the input folder
    # TODO: extend this to all non-Pd files
    CUSTOM_RENDER_SOURCE_PATH="$pdpath/heavy/render.cpp"
    if [ -f "$CUSTOM_RENDER_SOURCE_PATH" ]; then
        echo "Found custom heavy/render.cpp file in input folder, using that one instead of the default one.";
        cp "$CUSTOM_RENDER_SOURCE_PATH" "$projectpath/render.cpp" || exit 1
    else
        echo "Using default heavy/render.cpp"
        cp "$HVRESOURCES_DIR/heavy_render.cpp" "$projectpath/render.cpp" || exit 1
    fi
    
    echo "Updating files on board..."
    # HvContext* files tend to hang when transferring with rsync because they are very large and -c checksum takes a lot, I guess
    
    touch $reference_time_file
    # Transfer the files 
    rsync -ac --out-format="   %n" --no-t --delete-during --exclude='HvContext_'$ENZIENAUDIO_COM_PATCH_NAME'.*' --exclude=build --exclude=$BBB_PROJECT_NAME "$projectpath"/ "$BBB_NETWORK_TARGET_FOLDER" &&\
        { [ $NO_UPLOAD -eq 1 ] || scp "$projectpath"/HvContext* $BBB_NETWORK_TARGET_FOLDER; } ||\
	{ echo "ERROR: while synchronizing files with the BBB. Is the board connected?"; exit 1; }

    # TODO: rsync should upload a list of modified files, so that the corresponding objects can be deleted
    # TODO: this should be run only when Heavy_bela.h changes. Otherwise render is recompiled every time for no good reason
    #ssh $BBB_ADDRESS "rm -rf ${BBB_PROJECT_FOLDER}/build/render.*" 

    #produce a list of files which content has changed (not just the date)
    # remove old executable to force re-linking
    #if [ $NO_UPLOAD -eq 0 ]; then
    #    ssh $BBB_ADDRESS "rm -rf "$BBB_PROJECT_FOLDER/$BBB_PROJECT_NAME;
    #fi;
    # Make new Bela executable and run
    # It does not look very nice that we type the same things over and over
    # but that is because each line is an ssh session in its own right
    MAKE_COMMAND="make stop -C $BBB_BELA_HOME PROJECT='$BBB_PROJECT_NAME' CL='$COMMAND_ARGS'"
    if [ $RUN_PROJECT -eq 0 ]
    then
        echo "Building project..."
        ssh $BBB_ADDRESS "$MAKE_COMMAND"
    else
        echo "Building and running project..."
        if [ $WATCH -eq 1 ]
	then
	   # try to emulate run_without_screen: run with fifo
           if [ $RUN_WITHOUT_SCREEN -eq 1 ];
	   then 
	   	  ssh $BBB_ADDRESS "$MAKE_COMMAND runscreenfifo" & BACKGROUND_PROCESS_PID=$!
		  # run this in the background, it will be killed anyhow when the process stops.
		  # Either with the trap below or in another way
                 trap "kill $BACKGROUND_PROCESS_PID; exit 0;" 2 9 
		 # add the line below to the trap if you want to kill the process on the board when you exit the script
		 # ssh -o ConnectTimeout 2 $BBB_ADDRESS make --no-print-directory -C $BBB_BELA_HOME stop ;
	     else
                 ssh $BBB_ADDRESS "$MAKE_COMMAND runscreen"
	     fi
        elif [ $RUN_WITHOUT_SCREEN -eq 1 ]
        then
            ssh -t $BBB_ADDRESS "$MAKE_COMMAND run" 
        elif [ $RUN_IN_FOREGROUND -eq 1 ]
        then
            # Run in screen without detaching
            ssh -t $BBB_ADDRESS "$MAKE_COMMAND runscreenfg"
	else 
            # Run in screen and detach
            ssh $BBB_ADDRESS "$MAKE_COMMAND runscreen"
        fi
    fi
} #uploadBuildRun

uploadBuildRun

if [ $WATCH -ne 0 ]; then
	BACK_NO_UPLOAD=$NO_UPLOAD
	while true
	do
		# actually we are watching multiple paths : $pdpath and $HVRESOURCES_DIR
		# so that it is easier to edit hvresources code without risk of being 
		# overwritten, but we avoid mentioning it to the end user, otherwise they
		# get confused.
		echo "Waiting for changes in $pdpath, or press ctrl-c to terminate"
		while sleep 1
		do
			folder_has_changed "$pdpath" "$reference_time_file" && {
				echo "Content of $pdpath has changed"
				break
			}
			folder_has_changed "$HVRESOURCES_DIR" "$reference_time_file" && {
				echo "Content of $pdpath has changed"
				break
			}
		done
		echo "Files changed"
		# if .pd files did not change, no point in re-uploading
		folder_has_changed "$pdpath" "$reference_time_file" "\.pd" &&\
			NO_UPLOAD=$BACK_NO_UPLOAD || NO_UPLOAD=1
		uploadBuildRun
	done
fi;
