/*
 * render.cpp
 *
 *  Created on: Oct 24, 2014
 *      Author: parallels
 */


#include <BeagleRT.h>
#include <Utilities.h>
#include <cmath>

int gBufferSize = 8192;

// Buffers to hold samples for noise analysis
float* gBuffers[10];
int gBufferPointers[10];

// Outputs to display
float gDCLevels[10];
float gNoiseLevels[10];
float gNumSamplesAnalysed[10];

// Task to print results which would otherwise be too slow for render()
AuxiliaryTask gPrintTask;

void printResults();

// setup() is called once before the audio rendering starts.
// Use it to perform any initialisation and allocation which is dependent
// on the period size or sample rate.
//
// userData holds an opaque pointer to a data structure that was passed
// in from the call to initAudio().
//
// Return true on success; returning false halts the program.

bool setup(BeagleRTContext *context, void *userData)
{	
	// Clear the filter data structures
	for(int i = 0; i < 10; i++) {
		gBufferPointers[i] = 0;
		gBuffers[i] = new float[gBufferSize];
		if(gBuffers[i] == 0) {
			rt_printf("Error allocating buffer %d\n", i);
			return false;
		}
	}
	
	gPrintTask = BeagleRT_createAuxiliaryTask(printResults, 50, "beaglert-print-results");

	return true;
}

// render() is called regularly at the highest priority by the audio engine.
// Input and output are given from the audio hardware and the other
// ADCs and DACs (if available). If only audio is available, numMatrixFrames
// will be 0.

void render(BeagleRTContext *context, void *userData)
{
	bool bufferIsFull = false;	// Whether at least one buffer has filled
	
	for(unsigned int n = 0; n < context->audioFrames; n++) {
		// Store audio inputs in buffer
		for(unsigned int ch = 0; ch < context->audioChannels; ch++) {
			if(gBufferPointers[ch] < gBufferSize) {
				gBuffers[ch][gBufferPointers[ch]] = 
					context->audioIn[n * context->audioChannels + ch];
				gBufferPointers[ch]++;
				if(gBufferPointers[ch] >= gBufferSize)
					bufferIsFull = true;
			}
		}
	}
	
	if(context->analogChannels != 0) {
		for(unsigned int n = 0; n < context->analogFrames; n++) {
			// Store analog inputs in buffer, starting at channel 2
			for(unsigned int ch = 0; ch < context->analogChannels; ch++) {
				if(gBufferPointers[ch + 2] < gBufferSize) {
					gBuffers[ch + 2][gBufferPointers[ch + 2]] = 
						context->analogIn[n * context->analogChannels + ch];
					gBufferPointers[ch + 2]++;
					if(gBufferPointers[ch + 2] >= gBufferSize)
						bufferIsFull = true;
				}
			}
		}	
	}

	if(bufferIsFull) {
		// Analyse all active channels at once
		for(int ch = 0; ch < 10; ch++) {
			// gBufferPointers[ch] tells us how many samples were stored in the buffer
			gNumSamplesAnalysed[ch] = gBufferPointers[ch];
			
			if(gBufferPointers[ch] != 0) {
				float mean = 0;
				for(int n = 0; n < gBufferPointers[ch]; n++) {
					mean += gBuffers[ch][n];
				}
				mean /= (float)gBufferPointers[ch];
				
				float rms = 0;
				for(int n = 0; n < gBufferPointers[ch]; n++) {
					rms += (gBuffers[ch][n] - mean) * (gBuffers[ch][n] - mean);
				}				
				rms = sqrtf(rms / (float)gBufferPointers[ch]);
				
				gDCLevels[ch] = mean;
				gNoiseLevels[ch] = rms;
			}
			
			// Reset pointer to 0 for next time
			gBufferPointers[ch] = 0;
		}
		
		BeagleRT_scheduleAuxiliaryTask(gPrintTask);
	}
}

void printResults()
{
	rt_printf("\e[1;1H\e[2J");	// Command to clear the screen
	
	// Print the analysis results. channels 0-1 are audio, channels 2-9 are analog
	for(int ch = 0; ch < 10; ch++) {
		int samples = gNumSamplesAnalysed[ch];
		if(samples == 0)
			continue;
		
		if(ch == 0)
			rt_printf("Audio In L:  ");
		else if(ch == 1)
			rt_printf("Audio In R:  ");
		else
			rt_printf("Analog In %d: ", ch - 2);
		
		rt_printf("Noise %6.1fdB    DC offset %6.4f (%6.1fdB)    window size: %d\n", 
					20.0f * log10f(gNoiseLevels[ch]),
					gDCLevels[ch], 
					20.0f * log10f(fabsf(gDCLevels[ch])),
					samples);
	}
}

// cleanup() is called once at the end, after the audio has stopped.
// Release any resources that were allocated in setup().

void cleanup(BeagleRTContext *context, void *userData)
{
	for(int i = 0; i < 10; i++)
		delete gBuffers[i];
}
