#!/bin/bash
#
# This script enables or disables running BeagleRT when the board starts
# up.

BBB_ADDRESS="root@192.168.7.2"
BBB_PATH="/root/BeagleRT"
ENABLE_STARTUP=1
COMMAND_ARGS=
RUN_IN_LOOP=0

# This path is hard-coded in the BeagleRT image at present.
BBB_STARTUP_SCRIPT="/root/BeagleRT_startup.sh"

function usage
{
    THIS_SCRIPT=`basename "$0"`
    echo "Usage: $THIS_SCRIPT [-b path-on-beaglebone] [-c command-line-args] [-n] [-l]"

    echo "
    This script enables (by default) or disables running the BeagleRT
    project at startup. The -n option disables auto-startup, otherwise
    auto-startup is enabled. The -b option changes the default path, which
    is otherwise $BBB_PATH. The -c option passes command-line arguments to
    the BeagleRT program; enclose the argument string in quotes.
    The -l option runs the BeagleRT program in a loop, restarting
    automatically in the event of a crash."
}

OPTIND=1

while getopts "b:c:nlh" opt; do
    case $opt in
        b)            BBB_PATH=$OPTARG
                      ;;
        c)            COMMAND_ARGS=$OPTARG
                      ;;
        n)            ENABLE_STARTUP=0
                      ;;
        l)            RUN_IN_LOOP=1
                      ;;
        h|\?)         usage
                      exit 1
    esac
done

shift $((OPTIND-1))

if [ $ENABLE_STARTUP -eq 0 ]
then
    echo "Disabling BeagleRT at startup..."

    ssh $BBB_ADDRESS "echo \"#!/bin/sh
#
# This file is autogenerated by BeagleRT. Do not edit!

# Run on startup disabled -- nothing to do here\" > $BBB_STARTUP_SCRIPT"

else
    echo "Enabling BeagleRT at startup..."
  
    SCRIPT_PRERUN=
    SCRIPT_POSTRUN=
    if [ $RUN_IN_LOOP -ne 0 ] ; then
	SCRIPT_PRERUN="bash -c \\\"while true ; do"
	SCRIPT_POSTRUN=" ; done \\\" "
    fi

    ssh $BBB_ADDRESS "echo \"#!/bin/sh
#
# This file is autogenerated by BeagleRT. Do not edit!

echo Running BeagleRT...
screen -S BeagleRT -d -m $SCRIPT_PRERUN $BBB_PATH/BeagleRT $COMMAND_ARGS $SCRIPT_POSTRUN\" > $BBB_STARTUP_SCRIPT"

fi