/*
 * Utilities.cpp
 *
 *  Created on: Oct 27, 2014
 *      Author: parallels
 */

#include "../include/Utilities.h"

// analogReadFrame()
//
// Returns the value of the given analog input at the given frame number.
float analogReadFrame(BeagleRTContext *context, int frame, int channel) {
	return context->analogIn[frame * context->analogChannels + channel];
}

// analogWriteFrame()
//
// Sets a given channel to a value for the current frame and, if persistent outputs are
// enabled, for all subsequent frames
void analogWriteFrame(BeagleRTContext *context, int frame, int channel, float value) {
	if(context->flags & BEAGLERT_FLAG_ANALOG_OUTPUTS_PERSIST) {
		for(unsigned int f = frame; f < context->analogFrames; f++)
			context->analogOut[frame * context->analogChannels + channel] = value;
	}
	else
		context->analogOut[frame * context->analogChannels + channel] = value;
}

// analogWriteFrameOnce()
//
// Sets a given channel to a value for only the current frame
void analogWriteFrameOnce(BeagleRTContext *context, int frame, int channel, float value) {
	context->analogOut[frame * context->analogChannels + channel] = value;
}

// digitalReadFrame()
//
// Returns the value of a given digital input at the given frame number
int digitalReadFrame(BeagleRTContext *context, int frame, int channel) {
	return getBit(context->digital[frame], channel + 16);
}

// digitalWriteFrame()
//
// Sets a given digital output channel to a value for the current frame and all subsequent frames
void digitalWriteFrame(BeagleRTContext *context, int frame, int channel, int value) {
	for(unsigned int f = frame; f < context->digitalFrames; f++) {
		if(value)
			context->digital[f] |= 1 << (channel + 16);
		else
			context->digital[f] &= ~(1 << (channel + 16));
	}
}

// digitalWriteFrameOnce()
//
// Sets a given digital output channel to a value for the current frame only
void digitalWriteFrameOnce(BeagleRTContext *context, int frame, int channel, int value) {
	if(value)
		context->digital[frame] |= 1 << (channel + 16);
	else
		context->digital[frame] &= ~(1 << (channel + 16));
}

// pinModeFrame()
//
// Sets the direction of a digital pin for the current frame and all subsequent frames
void pinModeFrame(BeagleRTContext *context, int frame, int channel, int mode) {
	for(unsigned int f = frame; f < context->digitalFrames; f++) {
		if(mode)
			context->digital[f] |= (1 << channel);
		else
			context->digital[f] &= ~(1 << channel);
	}
}

// pinModeFrameOnce()
//
// Sets the direction of a digital pin for the current frame only
void pinModeFrameOnce(BeagleRTContext *context, int frame, int channel, int mode) {
	if(mode)
		context->digital[frame] |= (1 << channel);
	else
		context->digital[frame] &= ~(1 << channel);
}

// map()
//
// Scale an input value from one range to another. Works like its Wiring language equivalent.
// x is the value to scale; in_min and in_max are the input range; out_min and out_max
// are the output range.

float map(float x, float in_min, float in_max, float out_min, float out_max)
{
	return (x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
}

// constrain()
//
// Clips an input value to be between two end points
// x is the value to constrain; min_val and max_val are the range

float constrain(float x, float min_val, float max_val)
{
	if(x < min_val) return min_val;
	if(x > max_val) return max_val;
	return x;
}
