/*
 ____  _____ _        _    
| __ )| ____| |      / \   
|  _ \|  _| | |     / _ \  
| |_) | |___| |___ / ___ \ 
|____/|_____|_____/_/   \_\.io

 */

 /*
 * render.cpp
 *
 *  Created on: Oct 24, 2014
 *      Author: parallels
 */

/**
\example 1_basic_audio_analog_passthrough

Audio and analog passthrough: input to output
-----------------------------------------

This sketch demonstrates how to read from and write to the audio and analog input and output buffers.

In `render()` you'll see a nested for loop structure. You'll see this in all Bela projects. 
The first for loop cycles through `audioFrames`, the second through 
`audioChannels` (in this case left 0 and right 1).

You can access any information about current audio and sensor settings you can do the following: 
`context->name_of_item`. For example `context->audioChannels` returns current number of channels,
`context->audioFrames` returns the current number of audio frames, 
`context->audioSampleRate` returns the audio sample rate.

You can look at all the information you can access in ::BeagleRTContext.

Reading and writing from the audio buffers
------------------------------------------

The simplest way to read samples from the audio input buffer is with
`audioRead()` which we pass three arguments: context, current audio 
frame and current channel. In this example we have 
`audioRead(context, n, ch)` where both `n` and `ch` are provided by 
the nested for loop structure.

We can write samples to the audio output buffer in a similar way using 
`audioWrite()`. This has a fourth argument which is the value of the output.
For example `audioWrite(context, n, ch, value_to_output)`.

Reading and writing from the analog buffers
-------------------------------------------

The same is true for `analogRead()` and `analogWrite()`.

Note that for the analog channels we write to and read from the buffers in a separate set 
of nested for loops. This is because the they are sampled at half audio rate by default. 
The first of these for loops cycles through `analogFrames`, the second through
`analogChannels`.

By setting `audioWriteFrame(context, n, ch, audioReadFrame(context, n, ch))` and
`analogWrite(context, n, ch, analogReadFrame(context, n, ch))` we have a simple 
passthrough of audio input to output and analog input to output.


It is also possible to address the buffers directly, for example: 
`context->audioOut[n * context->audioChannels + ch]`.
*/

#include <Bela.h>
#include <rtdk.h>

// setup() is called once before the audio rendering starts.
// Use it to perform any initialisation and allocation which is dependent
// on the period size or sample rate.
//
// userData holds an opaque pointer to a data structure that was passed
// in from the call to initAudio().
//
// Return true on success; returning false halts the program.

bool setup(BelaContext *context, void *userData)
{
	// Nothing to do here...
	return true;
}

// render() is called regularly at the highest priority by the audio engine.
// Input and output are given from the audio hardware and the other
// ADCs and DACs (if available). If only audio is available, numMatrixFrames
// will be 0.

void render(BelaContext *context, void *userData)
{
	// Simplest possible case: pass inputs through to outputs
	for(unsigned int n = 0; n < context->audioFrames; n++) {
		for(unsigned int ch = 0; ch < context->audioChannels; ch++){
			// Two equivalent ways to write this code

			// The long way, using the buffers directly:
			// context->audioOut[n * context->audioChannels + ch] =
			// 		context->audioIn[n * context->audioChannels + ch];

			// Or using the macros:
			audioWrite(context, n, ch, audioRead(context, n, ch));
		}
	}

	// Same with analog channelss
	for(unsigned int n = 0; n < context->analogFrames; n++) {
		for(unsigned int ch = 0; ch < context->analogChannels; ch++) {
			// Two equivalent ways to write this code

			// The long way, using the buffers directly:
			// context->analogOut[n * context->analogChannels + ch] = context->analogIn[n * context->analogChannels + ch];

			// Or using the macros:
			analogWrite(context, n, ch, analogRead(context, n, ch));
		}
	}
}

// cleanup() is called once at the end, after the audio has stopped.
// Release any resources that were allocated in setup().

void cleanup(BelaContext *context, void *userData)
{

}
