/*
 * render.cpp
 *
 *  Created on: Oct 24, 2014
 *      Author: parallels
 */


#include "../../include/render.h"
#include "../../include/Utilities.h"
#include <rtdk.h>
#include <cmath>

float gPhase;
float gInverseSampleRate;
int gNumChannels;

// These settings are carried over from main.cpp
// Setting global variables is an alternative approach
// to passing a structure to userData in initialise_render()

extern int gSensorInputFrequency;
extern int gSensorInputAmplitude;

// initialise_render() is called once before the audio rendering starts.
// Use it to perform any initialisation and allocation which is dependent
// on the period size or sample rate.
//
// userData holds an opaque pointer to a data structure that was passed
// in from the call to initAudio().
//
// Return true on success; returning false halts the program.

bool initialise_render(int numChannels, int numMatrixFramesPerPeriod,
					   int numAudioFramesPerPeriod, float matrixSampleRate,
					   float audioSampleRate, void *userData)
{
	if(numMatrixFramesPerPeriod*2 != numAudioFramesPerPeriod) {
		rt_printf("Error: this example needs the matrix enabled, running at half audio rate\n");
		return false;
	}

	gNumChannels = numChannels;
	gInverseSampleRate = 1.0 / audioSampleRate;
	gPhase = 0.0;

	return true;
}

// render() is called regularly at the highest priority by the audio engine.
// Input and output are given from the audio hardware and the other
// ADCs and DACs (if available). If only audio is available, numMatrixFrames
// will be 0.

void render(int numMatrixFrames, int numAudioFrames, float *audioIn, float *audioOut,
			uint16_t *matrixIn, uint16_t *matrixOut)
{
	float frequency = 0;
	float amplitude = 0;

	// There are twice as many audio frames as matrix frames since audio sample rate
	// is twice as high

	for(int n = 0; n < numAudioFrames; n++) {
		if(!(n % 2)) {
			// Even audio samples: update frequency and amplitude from the matrix
			frequency = map((float)analogRead(gSensorInputFrequency, n/2), 0, MATRIX_MAX, 100, 1000);
			amplitude = (float)analogRead(gSensorInputAmplitude, n/2) / MATRIX_MAX;
		}

		float out = amplitude * sinf(gPhase);

		for(int channel = 0; channel < gNumChannels; channel++)
			audioOut[n * gNumChannels + channel] = out;

		gPhase += 2.0 * M_PI * frequency * gInverseSampleRate;
		if(gPhase > 2.0 * M_PI)
			gPhase -= 2.0 * M_PI;
	}
}

// cleanup_render() is called once at the end, after the audio has stopped.
// Release any resources that were allocated in initialise_render().

void cleanup_render()
{

}
