/*
 * render.cpp
 *
 *  Template render.cpp file for on-board heavy compiling
 *
 *  N.B. this is currently *not* compatible with foleyDesigner source files!
 *
 *  Created on: November 5, 2015
 *
 *  Christian Heinrichs
 *
 */

#include <Bela.h>
#include <Midi.h>
#include <cmath>
#include "Heavy_bbb.h"
#include <string.h>
#include <stdlib.h>
#include <string.h>
#include <DigitalChannelManager.h>

/*
 *	HEAVY CONTEXT & BUFFERS
 */

Hv_bbb *gHeavyContext;
float *gHvInputBuffers = NULL, *gHvOutputBuffers = NULL;
unsigned int gHvInputChannels = 0, gHvOutputChannels = 0;

float gInverseSampleRate;

/*
 *	HEAVY FUNCTIONS
 */

// TODO: rename this
#define LIBPD_DIGITAL_OFFSET 11 // digitals are preceded by 2 audio and 8 analogs (even if using a different number of analogs)

void printHook(double timestampSecs, const char *printLabel, const char *msgString, void *userData) {
  rt_printf("Message from Heavy patch: [@ %.3f] %s: %s\n", timestampSecs, printLabel, msgString);
}


// digitals
static DigitalChannelManager dcm;

void sendDigitalMessage(bool state, unsigned int delay, void* receiverName){
	hv_sendFloatToReceiver(gHeavyContext, hv_stringToHash((char*)receiverName), (float)state);
//	rt_printf("%s: %d\n", (char*)receiverName, state);
}

// TODO: turn them into hv hashes and adjust sendDigitalMessage accordingly
char hvDigitalInHashes[16][21]={
	{"bela_digitalIn11"},{"bela_digitalIn12"},{"bela_digitalIn13"},{"bela_digitalIn14"},{"bela_digitalIn15"},
	{"bela_digitalIn16"},{"bela_digitalIn17"},{"bela_digitalIn18"},{"bela_digitalIn19"},{"bela_digitalIn20"},
	{"bela_digitalIn21"},{"bela_digitalIn22"},{"bela_digitalIn23"},{"bela_digitalIn24"},{"bela_digitalIn25"},
	{"bela_digitalIn26"}
};

static void sendHook(
	double timestamp, // in milliseconds
	const char *receiverName,
	const HvMessage *const m,
	void *userData) {

	// Bela digital
	
	// Bela digital run-time messages

	// TODO: this first block is almost an exact copy of libpd's code, should we add this to the class?
	// let's make this as optimized as possible for built-in digital Out parsing
	// the built-in digital receivers are of the form "bela_digitalOutXX" where XX is between 11 and 26
	static int prefixLength = 15; // strlen("bela_digitalOut")
	if(strncmp(receiverName, "bela_digitalOut", prefixLength)==0){
		if(receiverName[prefixLength] != 0){ //the two ifs are used instead of if(strlen(source) >= prefixLength+2)
			if(receiverName[prefixLength + 1] != 0){
				// quickly convert the suffix to integer, assuming they are numbers, avoiding to call atoi
				int receiver = ((receiverName[prefixLength] - 48) * 10);
				receiver += (receiverName[prefixLength+1] - 48);
				unsigned int channel = receiver - LIBPD_DIGITAL_OFFSET; // go back to the actual Bela digital channel number
				bool value = hv_msg_getFloat(m, 0);
				if(channel < 16){ //16 is the hardcoded value for the number of digital channels
					dcm.setValue(channel, value);
				}
			}
		}
	}

	// Bela digital initialization messages
	if(strcmp(receiverName, "bela_setDigital") == 0){
		// Third argument (optional) can be ~ or sig for signal-rate, message-rate otherwise.
		// [in 14 ~(
		// |
		// [s bela_setDigital]
		// is signal("sig" or "~") or message("message", default) rate
		bool isMessageRate = true; // defaults to message rate
		bool direction = 0; // initialize it just to avoid the compiler's warning
		bool disable = false;
		int numArgs = hv_msg_getNumElements(m);
		if(numArgs < 2 || numArgs > 3 || !hv_msg_isSymbol(m, 0) || !hv_msg_isFloat(m, 1))
			return;
		if(numArgs == 3 && !hv_msg_isSymbol(m,2))
			return;
		char * symbol = hv_msg_getSymbol(m, 0);

		if(strcmp(symbol, "in") == 0){
			direction = INPUT;
		} else if(strcmp(symbol, "out") == 0){
			direction = OUTPUT;
		} else if(strcmp(symbol, "disable") == 0){
			disable = true;
		} else {
			return;
		}
		int channel = hv_msg_getFloat(m, 1) - LIBPD_DIGITAL_OFFSET;
		if(disable == true){
			dcm.unmanage(channel);
			return;
		}
		if(numArgs >= 3){
			char* s = hv_msg_getSymbol(m, 2);
			if(strcmp(s, "~") == 0  || strncmp(s, "sig", 3) == 0){
				isMessageRate = false;
			}
		}
		dcm.manage(channel, direction, isMessageRate);
	}
}


/*
 * SETUP, RENDER LOOP & CLEANUP
 */



// Midi
Midi midi;
unsigned int hvMidiHashes[7];


bool setup(BelaContext *context, void *userData)	{

	printf("top o setup\n");
	/* HEAVY */
	hvMidiHashes[kmmNoteOn] = hv_stringToHash("__hv_notein");
	hvMidiHashes[kmmNoteOff] = hv_stringToHash("noteoff"); // this is handled differently, see the render function
	hvMidiHashes[kmmControlChange] = hv_stringToHash("__hv_ctlin");
	hvMidiHashes[kmmProgramChange] = hv_stringToHash("pgmin");
	hvMidiHashes[kmmPolyphonicKeyPressure] = hv_stringToHash("polytouchin");
	hvMidiHashes[kmmChannelPressure] = hv_stringToHash("touchin");
	hvMidiHashes[kmmPitchBend] = hv_stringToHash("bendin");

	printf("after midi o setup\n");
	gHeavyContext = hv_bbb_new(context->audioSampleRate);

	printf("aftet new  o setup\n");
	gHvInputChannels = hv_getNumInputChannels(gHeavyContext);
	gHvOutputChannels = hv_getNumOutputChannels(gHeavyContext);

	rt_printf("Starting Heavy context with %d input channels and %d output channels\n",
			  gHvInputChannels, gHvOutputChannels);

	if(gHvInputChannels != 0) {
		gHvInputBuffers = (float *)calloc(gHvInputChannels * context->audioFrames,sizeof(float));
	}
	if(gHvOutputChannels != 0) {
		gHvOutputBuffers = (float *)calloc(gHvOutputChannels * context->audioFrames,sizeof(float));
	}
	printf("mid o setup\n");

	gInverseSampleRate = 1.0 / context->audioSampleRate;

	// Set heavy print hook
	hv_setPrintHook(gHeavyContext, printHook);
	// Set heavy send hook
	hv_setSendHook(gHeavyContext, sendHook);

	// TODO: change these hardcoded port values and actually change them in the Midi class
	midi.readFrom(0);
	midi.writeTo(0);
	midi.enableParser(true);

	// Bela digital
	dcm.setCallback(sendDigitalMessage);
	if(context->digitalChannels > 0){
		for(unsigned int ch = 0; ch < context->digitalChannels; ++ch){
			dcm.setCallbackArgument(ch, hvDigitalInHashes[ch]);
		}
	}
	// unlike libpd, no need here to bind the bela_digitalOut.. receivers

	printf("end o setup\n");
	return true;
}


void render(BelaContext *context, void *userData)
{
	{
		int num;
		while((num = midi.getParser()->numAvailableMessages()) > 0){
			static MidiChannelMessage message;
			message = midi.getParser()->getNextChannelMessage();
			switch(message.getType()){
			case kmmNoteOn: {
				// message.prettyPrint();
				float noteNumber = message.getDataByte(0);
				float velocity = message.getDataByte(1);
				float channel = message.getChannel();
				// rt_printf("message: noteNumber: %f, velocity: %f, channel: %f\n", noteNumber, velocity, channel);
				hv_vscheduleMessageForReceiver(gHeavyContext, hvMidiHashes[kmmNoteOn], 0, "fff", noteNumber, velocity, channel);
				break;
			}
			case kmmNoteOff:
			{
				/* PureData does not seem to handle noteoff messages as per the MIDI specs,
				 * so that the noteoff velocity is ignored. Here we convert them to noteon
				 * with a velocity of 0.
				 */
				float noteNumber = message.getDataByte(0);
				// int velocity = message.getDataByte(1); // would be ignored by Pd
				float channel = message.getChannel();
				// note we are sending the below to hvHashes[kmmNoteOn] !!
				hv_vscheduleMessageForReceiver(gHeavyContext, hvMidiHashes[kmmNoteOn], 0, "fff", noteNumber, 0, channel);
				break;
			}
			case kmmControlChange: {
				int channel = message.getChannel();
				int controller = message.getDataByte(0);
				int value = message.getDataByte(1);
				//TODO: maybe the order of the arguments is wrong here?
				hv_vscheduleMessageForReceiver(gHeavyContext, hvMidiHashes[kmmControlChange], 0, "fff",
						value, controller, channel);
				break;
			}
			case kmmProgramChange: {
				int channel = message.getChannel();
				int program = message.getDataByte(0);
				hv_vscheduleMessageForReceiver(gHeavyContext, hvMidiHashes[kmmProgramChange], 0, "ff",
						program, channel);
				//TODO: maybe the order of the arguments is wrong here?
				break;
			}
			case kmmPolyphonicKeyPressure: {
				int channel = message.getChannel();
				int pitch = message.getDataByte(0);
				int value = message.getDataByte(1);
				//TODO: maybe the order of the arguments is wrong here?
				hv_vscheduleMessageForReceiver(gHeavyContext, hvMidiHashes[kmmPolyphonicKeyPressure], 0, "fff",
						channel, pitch, value);
				break;
			}
			case kmmChannelPressure:
			{
				int channel = message.getChannel();
				int value = message.getDataByte(0);
				//TODO: maybe the order of the arguments is wrong here?
				hv_vscheduleMessageForReceiver(gHeavyContext, hvMidiHashes[kmmChannelPressure], 0, "ff",
						channel, value);
				break;
			}
			case kmmPitchBend:
			{
				int channel = message.getChannel();
				int value = ((message.getDataByte(1) << 7) | message.getDataByte(0)) + 8192;
				//TODO: is the value range correct?
				//TODO: maybe the order of the arguments is wrong here?
				hv_vscheduleMessageForReceiver(gHeavyContext, hvMidiHashes[kmmPitchBend], 0, "ff",
						channel, value);
				break;
			}
			case kmmNone:
			case kmmAny:
				break;
			}
		}
	}

	// De-interleave the data
	if(gHvInputBuffers != NULL) {
		for(unsigned int n = 0; n < context->audioFrames; n++) {
			for(unsigned int ch = 0; ch < gHvInputChannels; ch++) {
				if(ch >= context->audioChannels+context->analogChannels) {
					// THESE ARE PARAMETER INPUT 'CHANNELS' USED FOR ROUTING
					// 'sensor' outputs from routing channels of dac~ are passed through here
					break;
				} else {
					// If more than 2 ADC inputs are used in the pd patch, route the analog inputs
					// i.e. ADC3->analogIn0 etc. (first two are always audio inputs)
					if(ch >= context->audioChannels)	{
						int m = n/2;
						float mIn = context->analogIn[m*context->analogChannels + (ch-context->audioChannels)];
						gHvInputBuffers[ch * context->audioFrames + n] = mIn;
					} else {
						gHvInputBuffers[ch * context->audioFrames + n] = context->audioIn[n * context->audioChannels + ch];
					}
				}
			}
		}
	}

	// Bela digital in
	// note: in multiple places below we assume that the number of digital frames is same as number of audio
	// digital in at message-rate
	dcm.processInput(context->digital, context->digitalFrames);

	// Bela digital in at signal-rate
	// TODO: not really straightforward to implement as Heavy determines the number of channels in use at compile time
	// on the basis of the number of adc~ / dac~ of the patch ... Maybe we should always include
	// a dummy [adc~ 27] [dac~ 27] to make sure all channels are always allocated and then leave them all unprocessed ?

	// replacement for bang~ object
	//hv_vscheduleMessageForReceiver(gHeavyContext, "bbb_bang", 0.0f, "b");

	hv_bbb_process_inline(gHeavyContext, gHvInputBuffers, gHvOutputBuffers, context->audioFrames);

	// Bela digital out
	// digital out at signal-rate
	// TODO: see note above.

	// digital out at message-rate
	dcm.processOutput(context->digital, context->digitalFrames);

	// Interleave the output data
	if(gHvOutputBuffers != NULL) {
		for(unsigned int n = 0; n < context->audioFrames; n++) {

			for(unsigned int ch = 0; ch < gHvOutputChannels; ch++) {
				if(ch >= context->audioChannels+context->analogChannels) {
					// THESE ARE SENSOR OUTPUT 'CHANNELS' USED FOR ROUTING
					// they are the content of the 'sensor output' dac~ channels
				} else {
					if(ch >= context->audioChannels)	{
						int m = n/2;
						context->analogOut[m * context->analogFrames + (ch-context->audioChannels)] = constrain(gHvOutputBuffers[ch*context->audioFrames + n],0.0,1.0);
					} else {
						context->audioOut[n * context->audioChannels + ch] = gHvOutputBuffers[ch * context->audioFrames + n];
					}
				}
			}
		}
	}

}


void cleanup(BelaContext *context, void *userData)
{

	hv_bbb_free(gHeavyContext);
	if(gHvInputBuffers != NULL)
		free(gHvInputBuffers);
	if(gHvOutputBuffers != NULL)
		free(gHvOutputBuffers);

}
