#!/bin/sh
#
# This script compiles a Bela project on the BeagleBone Black and
# optionally runs it. Pass a directory path in the first argument. 
# The source files in this directory are copied to the board and compiled.

# set defaults unless variables are already set
[ -z "$BBB_ADDRESS" ] && BBB_ADDRESS="root@192.168.7.2"
[ -z "$BBB_BELA_HOME" ] && BBB_BELA_HOME="~/Bela/"
[ -z "$BBB_SCREEN_NAME" ] && BBB_SCREEN_NAME="Bela"
[ -z "$RUN_PROJECT" ] && RUN_PROJECT=1
[ -z "$COMMAND_ARGS" ] && COMMAND_ARGS=
[ -z "$RUN_IN_FOREGROUND" ] && RUN_IN_FOREGROUND=1
[ -z "$RUN_WITHOUT_SCREEN" ] && RUN_WITHOUT_SCREEN=0
[ -z "$BBB_PROJECT_HOME" ] && BBB_PROJECT_HOME="${BBB_BELA_HOME}/projects/"
[ -z "$BBB_DEFAULT_PROJECT_NAME" ] && BBB_DEFAULT_PROJECT_NAME="scriptUploadedProject"
[ -z "$BBB_PROJECT_NAME" ] && BBB_PROJECT_NAME=$BBB_DEFAULT_PROJECT_NAME

usage()
{
    THIS_SCRIPT=`basename "$0"`
    echo "Usage: $THIS_SCRIPT [-c command-line-args] [-nbfF] <directory-with-source-files>"
    echo "
    This script copies a directory of source files to the BeagleBone, compiles
    and runs it. The Bela core files should have first been copied over
    using the setup_board.sh script supplied with Bela.

    The source directory should contain at least one .c, .cpp or .S file.
    If the argument -n is passed, the output will not be run after compiling.
    The -c option passes command-line arguments to
    the Bela program; enclose the argument string in quotes.
	
    -p arg : sets the name of the project to run (default: $BBB_PROJECT_NAME ) 

    By default, the project runs in the foreground of the current terminal,
    within a screen session that can be detached later. The -f argument runs
    the project in the foreground of the current terminal, without screen, so
    the output can be piped to another destination. The -b argument runs it
    in a screen in the background, so no output is shown. The -m argument allows
    to pass arguments to the Makefile before the run target. For instance,
    pass -m \`"projectclean"\` or \`-m "distclean"\` to clean project-specific
    pre-built objects, or all the pre-built objects, respectively."
}

OPTIND=1

while getopts "bc:m:nfFhp:" opt; do
    case $opt in
        c)            COMMAND_ARGS=$OPTARG
                      ;;
        b)            RUN_IN_FOREGROUND=0
                      ;;
        f)            RUN_WITHOUT_SCREEN=1
                      ;;
        n)    	      RUN_PROJECT=0
                      ;;
	p)            BBB_PROJECT_NAME=$OPTARG
		      ;;	
	m)            BBB_MAKEFILE_OPTIONS=$OPTARG
	              ;;
        h|\?)         usage
                      exit 1
    esac
done

shift $((OPTIND-1))

# Check that we have a directory containing at least one source file
# as an argument

if [ -z "$1" ]
then
    usage
    exit
fi

FIND_STRING="find $* -maxdepth 10000 -type f "

C_FILES=$($FIND_STRING -name "*.c")
CPP_FILES=$($FIND_STRING -name "*.cpp")
ASM_FILES=$($FIND_STRING -name "*.S")

if [[ -z $C_FILES ]] && [[ -z $CPP_FILES ]] && [[ -z $ASM_FILES ]]
then
    echo "Please provide a directory containing .c, .cpp or .S files."
#    echo "Usage: $THIS_SCRIPT [directory-with-source-files]"
    usage
    exit
fi

BBB_PROJECT_FOLDER=$BBB_PROJECT_HOME"/"$BBB_PROJECT_NAME #make sure there is no trailing slash here
BBB_NETWORK_TARGET_FOLDER=$BBB_ADDRESS:$BBB_PROJECT_FOLDER

echo "Stopping running process..."
# sets the date and stop running process
ssh $BBB_ADDRESS "date -s '`date`' > /dev/null; mkdir -p $BBB_PROJECT_FOLDER; make -C $BBB_BELA_HOME stop"

#concatenate arguments to form path.
HOST_SOURCE_PATH= #initially empty, will be filled with input arguments
for i in "$@" #parse input arguments
do
  HOST_SOURCE_PATH+=" $1"
  shift
  # Copy new souce files to the board
done

# Copy new source files to the board
echo "Copying new source files to BeagleBone..."
if [ -z `which rsync` ];
then
    #if rsync is not available, brutally clean the destination folder
    #and copy over all the files again and recompile them
    ssh bbb "make -C $BBB_BELA_HOME sourceclean PROJECT=$BBB_PROJECT_NAME";
    scp $HOST_SOURCE_PATH "$BBB_NETWORK_TARGET_FOLDER"
else
    #rsync 
    # --delete makes sure it removes files that are not in the origin folder
    # -c evaluates changes using md5 checksum instead of file date, so we don't care about time skews 
    # --no-t makes sure file timestamps are not preserved, so that the Makefile will not think that targets are up to date when replacing files on the BBB
    #  with older files from the host. This will solve 99% of the issues with Makefile thinking a target is up to date when it is not.

   echo rsync -avc --no-t --delete-after --exclude=$BBB_PROJECT_NAME --exclude=build $HOST_SOURCE_PATH"/" "$BBB_NETWORK_TARGET_FOLDER/" #trailing slashes used here make sure rsync does not create another folder inside the target folder
    rsync -avc --no-t --delete-after --exclude=$BBB_PROJECT_NAME --exclude=build $HOST_SOURCE_PATH"/" "$BBB_NETWORK_TARGET_FOLDER/" #trailing slashes used here make sure rsync does not create another folder inside the target folder
fi;

if [ $? -ne 0 ]
then
	echo "Error while copying files"
	exit
fi

# Make new Bela executable and run
MAKE_COMMAND="make -C $BBB_BELA_HOME PROJECT='$BBB_PROJECT_NAME' CL='$COMMAND_ARGS' $BBB_MAKEFILE_OPTIONS"
if [ $RUN_PROJECT -eq 0 ]
then
    echo "Building project..."
    ssh $BBB_ADDRESS "$MAKE_COMMAND"
else
    echo "Building and running project..."
	
	if [ $RUN_WITHOUT_SCREEN -ne 0 ]
	then
	    ssh -t $BBB_ADDRESS "$MAKE_COMMAND run"	
	elif [ $RUN_IN_FOREGROUND -eq 0 ]
	then
	    ssh $BBB_ADDRESS "$MAKE_COMMAND runscreen"
	else
	    ssh -t $BBB_ADDRESS "$MAKE_COMMAND runscreenfg"
	fi
fi
