#!/bin/bash
#
# This script compiles a BeagleRT project on the BeagleBone Black and
# optionally runs it. Pass a directory path in the first argument. 
# The source files in this directory are copied to the board and compiled.

# set defaults unless variables are already set
[ -z "$BBB_ADDRESS" ] && BBB_ADDRESS="root@192.168.7.2"
[ -z "$BBB_BELA_HOME" ] && BBB_BELA_HOME="~/BeagleRT/"
[ -z "$BBB_SCREEN_NAME" ] && BBB_SCREEN_NAME="BeagleRT"
[ -z "$RUN_PROJECT" ] && RUN_PROJECT=1
[ -z "$COMMAND_ARGS" ] && COMMAND_ARGS=
[ -z "$RUN_IN_FOREGROUND" ] && RUN_IN_FOREGROUND=1
[ -z "$RUN_WITHOUT_SCREEN" ] && RUN_WITHOUT_SCREEN=0
[ -z "$BBB_PROJECT_HOME" ] && BBB_PROJECT_HOME="${BBB_BELA_HOME}/projects/"
[ -z "$BBB_PROJECT_NAME" ] && BBB_PROJECT_NAME="scriptUploadedProject"

function usage
{
    THIS_SCRIPT=`basename "$0"`
    echo "Usage: $THIS_SCRIPT [-b path-on-beaglebone] [-c command-line-args] [-nfF] <directory-with-source-files>"
    echo "
    This script copies a directory of source files to the BeagleBone, compiles
    and runs it. The BeagleRT core files should have first been copied over
    using the setup_board.sh script supplied with BeagleRT.

    The source directory should contain at least one .c, .cpp or .S file.
    If the argument -n is passed, the output will not be run after compiling.
    The argument -b will change the local path on the BeagleBone where the
    BeagleRT files are found. The -c option passes command-line arguments to
    the BeagleRT program; enclose the argument string in quotes.
	
    By default, the project runs in the foreground of the current terminal,
    within a screen session that can be detached later. The -F argument runs
    the project in the foreground of the current terminal, without screen, so
    the output can be piped to another destination. The -f argument runs it
    in a screen in the background, so no output is shown."
}

OPTIND=1

while getopts "b:c:nfFh" opt; do
    case $opt in
        b)            BBB_BELA_HOME=$OPTARG
                      ;;
        c)            COMMAND_ARGS=$OPTARG
                      ;;
        f)            RUN_IN_FOREGROUND=0
			          ;;
        F)            RUN_WITHOUT_SCREEN=1
			  		  ;;
        n)    	      RUN_PROJECT=0
                      ;;
        h|\?)         usage
                      exit 1
    esac
done

shift $((OPTIND-1))

# Check that we have a directory containing at least one source file
# as an argument

if [ -z "$1" ]
then
    usage
    exit
fi

FIND_STRING="find $* -maxdepth 10000 -type f "

C_FILES=$($FIND_STRING -name "*.c")
CPP_FILES=$($FIND_STRING -name "*.cpp")
ASM_FILES=$($FIND_STRING -name "*.S")

if [[ -z $C_FILES ]] && [[ -z $CPP_FILES ]] && [[ -z $ASM_FILES ]]
then
    echo "Please provide a directory containing .c, .cpp or .S files."
#    echo "Usage: $THIS_SCRIPT [directory-with-source-files]"
    usage
    exit
fi

BBB_PROJECT_FOLDER=$BBB_PROJECT_HOME"/"$BBB_PROJECT_NAME"/"
BBB_NETWORK_TARGET_FOLDER=$BBB_ADDRESS:$BBB_PROJECT_FOLDER

# Stop BeagleRT and clean out old source files
echo "Stopping running program..."
# sets the date, stops the running process
ssh $BBB_ADDRESS "date -s '`date`' > /dev/null; mkdir -p $BBB_PROJECT_FOLDER; screen -X -S '"$BBB_SCREEN_NAME"' quit &>/dev/null;"

#concatenate arguments to form path.
HOST_SOURCE_PATH= #initially empty, will be filled with input arguments
for i in "$@" #parse input arguments
do
  HOST_SOURCE_PATH+=" $1 "
  shift
  # Copy new souce files to the board
done

# Copy new source files to the board
echo "Copying new source files to BeagleBone..."
if [ -z `which rsync` ];
then
    #if rsync is not available, brutally clean the destination folder
    #and copy over all the files again and recompile them
    ssh bbb "make -C $BBB_BELA_HOME sourceclean PROJECT=$BBB_PROJECT_NAME";
    scp $HOST_SOURCE_PATH "$BBB_NETWORK_TARGET_FOLDER"
else
    #rsync --delete makes sure it removes files that are not in the origin folder
    rsync -av --delete-after --exclude=build $HOST_SOURCE_PATH "$BBB_NETWORK_TARGET_FOLDER"
fi;

if [ $? -ne 0 ]
then
	echo "Error while copying files"
	exit
fi

# Make new BeagleRT executable and run
if [ $RUN_PROJECT -eq 0 ]
then
    echo "Building project..."
    ssh $BBB_ADDRESS "make all -C $BBB_BELA_HOME PROJECT=$BBB_PROJECT_NAME"
else
    echo "Building and running project..."
	
	if [ $RUN_WITHOUT_SCREEN -ne 0 ]
	then
		ssh -t $BBB_ADDRESS "cd $BBB_BELA_HOME && make all && cd source && $BBB_PROJECT_FOLDER/$BBB_PROJECT_NAME $COMMAND_ARGS"	
	elif [ $RUN_IN_FOREGROUND -eq 0 ]
	then
	    ssh $BBB_ADDRESS "cd $BBB_BELA_HOME && make all PROJECT=$BBB_PROJECT_NAME && cd source && screen -S $BBB_SCREEN_NAME -d -m $BBB_PROJECT_FOLDER/$BBB_PROJECT_NAME $COMMAND_ARGS"
	else
	    ssh -t $BBB_ADDRESS "cd $BBB_BELA_HOME && make all PROJECT=$BBB_PROJECT_NAME && cd source && screen -S $BBB_SCREEN_NAME $BBB_PROJECT_FOLDER/$BBB_PROJECT_NAME $COMMAND_ARGS"
	fi
fi
