/*
 * render.cpp
 *
 *  Created on: Oct 24, 2014
 *      Author: parallels
 */


#include "../../include/render.h"
#include "../../include/Utilities.h"
#include <rtdk.h>
#include <cmath>

// Set range for analog outputs designed for driving LEDs
const float kMinimumAmplitude = (1.5 / 5.0) * MATRIX_MAX;
const float kAmplitudeRange = MATRIX_MAX - kMinimumAmplitude;

float gFrequency;
float gPhase;
float gInverseSampleRate;

// initialise_render() is called once before the audio rendering starts.
// Use it to perform any initialisation and allocation which is dependent
// on the period size or sample rate.
//
// userData holds an opaque pointer to a data structure that was passed
// in from the call to initAudio().
//
// Return true on success; returning false halts the program.

bool initialise_render(int numChannels, int numMatrixFramesPerPeriod,
					   int numAudioFramesPerPeriod, float matrixSampleRate,
					   float audioSampleRate, void *userData)
{
	// Retrieve a parameter passed in from the initAudio() call
	gFrequency = *(float *)userData;

	if(numMatrixFramesPerPeriod*2 != numAudioFramesPerPeriod) {
		rt_printf("Error: this example needs the matrix enabled, running at half audio rate\n");
		return false;
	}

	gInverseSampleRate = 1.0 / audioSampleRate;
	gPhase = 0.0;

	return true;
}

// render() is called regularly at the highest priority by the audio engine.
// Input and output are given from the audio hardware and the other
// ADCs and DACs (if available). If only audio is available, numMatrixFrames
// will be 0.

void render(int numMatrixFrames, int numAudioFrames, float *audioIn, float *audioOut,
			uint16_t *matrixIn, uint16_t *matrixOut)
{
	for(int n = 0; n < numMatrixFrames; n++) {
		// Set LED to different phase for each matrix channel
		float relativePhase = 0.0;
		for(int channel = 0; channel < 8; channel++) {
			float out = kMinimumAmplitude + kAmplitudeRange * 0.5f * (1.0f + sinf(gPhase + relativePhase));
			if(out > MATRIX_MAX)
				out = MATRIX_MAX;

			analogWrite(channel, n, out);

			// Advance by pi/4 (1/8 of a full rotation) for each channel
			relativePhase += M_PI * 0.25;
		}

		gPhase += 2.0 * M_PI * gFrequency * gInverseSampleRate;
		if(gPhase > 2.0 * M_PI)
			gPhase -= 2.0 * M_PI;
	}
}

// cleanup_render() is called once at the end, after the audio has stopped.
// Release any resources that were allocated in initialise_render().

void cleanup_render()
{

}
