/*
 * main.cpp
 *
 *  Created on: Oct 24, 2014
 *      Author: parallels
 */

#include <iostream>
#include <cstdlib>
#include <libgen.h>
#include <signal.h>
#include "../../include/RTAudio.h"

using namespace std;

// Handle Ctrl-C by requesting that the audio rendering stop
void interrupt_handler(int var)
{
	gShouldStop = true;
}

// Print usage information
void usage(const char * processName)
{
	cerr << "Usage: " << processName << " [-h] [-v] [-p period] [-f frequency]" << endl;
	cerr << "   -h:           Print this menu\n";
	cerr << "   -v:           Enable verbose messages\n";
	cerr << "   -p period:    Set the period (hardware buffer) size in sensor frames\n";
	cerr << "   -f frequency: Set the frequency of the oscillator\n";
	cerr << "   -m:           Enable the matrix (ADC and DAC) as well as audio\n";
}

int main(int argc, char *argv[])
{
	int periodSize = 8;			// Period size in sensor frames
	int verbose = 0;			// Verbose printing level
	float frequency = 440.0;	// Frequency of oscillator
	int useMatrix = 0;			// Whether to use the matrix or just audio

	// Parse command-line arguments
	while (1) {
		int c;
		if ((c = getopt(argc, argv, "hp:vf:m")) < 0)
				break;
		switch (c) {
		case 'h':
				usage(basename(argv[0]));
				exit(0);
		case 'p':
				periodSize = atoi(optarg);
				if(periodSize < 1)
					periodSize = 1;
				break;
		case 'v':
				verbose = 1;
				break;
		case 'f':
				frequency = atof(optarg);
				break;
		case 'm':
				useMatrix = 1;
				break;
		case '?':
		default:
				usage(basename(argv[0]));
				exit(1);
		}
	}


	// Set verbose logging information (optional by using value > 0; default is 0)
	setVerboseLevel(verbose);

	if(verbose) {
		cout << "Starting with period size " << periodSize << " and frequency " << frequency << endl;
		if(useMatrix)
			cout << "Matrix enabled\n";
		else
			cout << "Matrix disabled\n";
	}

	// Initialise the PRU audio device
	if(initAudio(periodSize, useMatrix, &frequency) != 0) {
		cout << "Error: unable to initialise audio" << endl;
		return -1;
	}

	// Start the audio device running
	if(startAudio()) {
		cout << "Error: unable to start real-time audio" << endl;
		return -1;
	}

	// Set up interrupt handler to catch Control-C
	signal(SIGINT, interrupt_handler);

	// Run until told to stop
	while(!gShouldStop) {
		usleep(100000);
	}

	// Stop the audio device
	stopAudio();

	if(verbose) {
		cout << "Cleaning up..." << endl;
	}

	// Clean up any resources allocated for audio
	cleanupAudio();

	// All done!
	return 0;
}
